<?php
/**
 * Build our admin dashboard.
 *
 * @package SekhloPress Premium
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * This class adds premium sections to our Dashboard.
 */
class SekhloPress_Pro_Dashboard {
	/**
	 * Class instance.
	 *
	 * @access private
	 * @var $instance Class instance.
	 */
	private static $instance;

	/**
	 * Initiator
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Get started.
	 */
	public function __construct() {
		add_action( 'after_setup_theme', array( $this, 'setup' ) );
	}

	/**
	 * Add our actions and require old Dashboard files if we need them.
	 */
	public function setup() {
		// Load our old dashboard if we're using an old version of SekhloPress.
		if ( ! class_exists( 'SekhloPress_Dashboard' ) ) {
			if ( is_admin() ) {
				require_once GP_PREMIUM_DIR_PATH . 'inc/legacy/dashboard.php';
				require_once GP_PREMIUM_DIR_PATH . 'inc/legacy/import-export.php';
				require_once GP_PREMIUM_DIR_PATH . 'inc/legacy/reset.php';
				require_once GP_PREMIUM_DIR_PATH . 'inc/legacy/activation.php';
			}

			return;
		}

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'sekhlo_admin_dashboard', array( $this, 'module_list' ), 8 );
		add_action( 'sekhlo_admin_dashboard', array( $this, 'license_key' ), 5 );
		add_action( 'sekhlo_admin_dashboard', array( $this, 'import_export' ), 50 );
		add_action( 'sekhlo_admin_dashboard', array( $this, 'reset' ), 100 );
		add_filter( 'sekhlo_premium_beta_tester', array( $this, 'set_beta_tester' ) );
	}

	/**
	 * Get data for all of our pro modules.
	 */
	public static function get_modules() {
		$modules = array(
			'Backgrounds' => array(
				'title' => __( 'Backgrounds', 'sekhlopress-connect' ),
				'description' => __( 'Set background images for various HTML elements.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_backgrounds',
				'settings' => 'sekhlo_background_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_backgrounds', false ),
				'exportable' => true,
			),
			'Blog' => array(
				'title' => __( 'Blog', 'sekhlopress-connect' ),
				'description' => __( 'Set blog options like infinite scroll, masonry layouts and more.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_blog',
				'settings' => 'sekhlo_blog_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_blog', false ),
				'exportable' => true,
			),
			'Colors' => array(
				'title' => __( 'Colors', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_colors',
				'isActive' => 'activated' === get_option( 'sekhlo_package_colors', false ),
			),
			'Copyright' => array(
				'title' => __( 'Copyright', 'sekhlopress-connect' ),
				'description' => __( 'Set a custom copyright message in your footer.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_copyright',
				'settings' => 'copyright',
				'isActive' => 'activated' === get_option( 'sekhlo_package_copyright', false ),
				'exportable' => true,
			),
			'Disable Elements' => array(
				'title' => __( 'Disable Elements', 'sekhlopress-connect' ),
				'description' => __( 'Disable default theme elements on specific pages or inside a Layout Element.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_disable_elements',
				'isActive' => 'activated' === get_option( 'sekhlo_package_disable_elements', false ),
			),
			'Elements' => array(
				'title' => __( 'Elements', 'sekhlopress-connect' ),
				'description' => __( 'Use our block editor theme builder, build advanced HTML hooks, and gain more Layout control.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_elements',
				'isActive' => 'activated' === get_option( 'sekhlo_package_elements', false ),
			),
			'Font Library' => array(
				'title' => __( 'Font Library', 'sekhlopress-connect' ),
				'description' => __( 'Download and localize fonts from the Google Fonts library.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_font_library',
				'isActive' => 'activated' === get_option( 'sekhlo_package_font_library', false ),
			),
			'Hooks' => array(
				'title' => __( 'Hooks', 'sekhlopress-connect' ),
				'description' => __( 'This module has been deprecated. Please use Elements instead.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_hooks',
				'settings' => 'sekhlo_hooks',
				'isActive' => 'activated' === get_option( 'sekhlo_package_hooks', false ),
				'exportable' => true,
			),
			'Menu Plus' => array(
				'title' => __( 'Menu Plus', 'sekhlopress-connect' ),
				'description' => __( 'Set up a mobile header, sticky navigation or off-canvas panel.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_menu_plus',
				'settings' => 'sekhlo_menu_plus_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_menu_plus', false ),
				'exportable' => true,
			),
			'Page Header' => array(
				'title' => __( 'Page Header', 'sekhlopress-connect' ),
				'description' => __( 'This module has been deprecated. Please use Elements instead.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_page_header',
				'settings' => 'sekhlo_page_header_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_page_header', false ),
				'exportable' => true,
			),
			'Secondary Nav' => array(
				'title' => __( 'Secondary Nav', 'sekhlopress-connect' ),
				'description' => __( 'Add a fully-featured secondary navigation to your site.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_secondary_nav',
				'settings' => 'sekhlo_secondary_nav_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_secondary_nav', false ),
				'exportable' => true,
			),
			'Sections' => array(
				'title' => __( 'Sections', 'sekhlopress-connect' ),
				'description' => __( 'This module has been deprecated. Please consider using our GenerateBlocks plugin instead.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_sections',
				'isActive' => 'activated' === get_option( 'sekhlo_package_sections', false ),
			),
			'Spacing' => array(
				'title' => __( 'Spacing', 'sekhlopress-connect' ),
				'description' => __( 'Set the padding and overall spacing of your theme elements.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_spacing',
				'settings' => 'sekhlo_spacing_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_spacing', false ),
				'exportable' => true,
			),
			'Typography' => array(
				'title' => __( 'Typography', 'sekhlopress-connect' ),
				'description' => __( 'This module has been deprecated. Switch to our dynamic typography system in Customize > General instead.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_typography',
				'isActive' => 'activated' === get_option( 'sekhlo_package_typography', false ),
			),
			'WooCommerce' => array(
				'title' => __( 'WooCommerce', 'sekhlopress-connect' ),
				'description' => __( 'Add colors, typography, and layout options to your WooCommerce store.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_woocommerce',
				'settings' => 'sekhlo_woocommerce_settings',
				'isActive' => 'activated' === get_option( 'sekhlo_package_woocommerce', false ),
				'exportable' => true,
			),
		);

		if ( version_compare( PHP_VERSION, '5.4', '>=' ) && ! defined( 'sekhlo_DISABLE_SITE_LIBRARY' ) ) {
			$modules['Site Library'] = array(
				'title' => __( 'Site Library', 'sekhlopress-connect' ),
				'description' => __( 'Choose from an extensive library of professionally designed starter sites.', 'sekhlopress-connect' ),
				'key' => 'sekhlo_package_site_library',
				'isActive' => 'activated' === get_option( 'sekhlo_package_site_library', false ),
			);
		}

		if ( function_exists( 'sekhlo_is_using_dynamic_typography' ) && sekhlo_is_using_dynamic_typography() ) {
			unset( $modules['Typography'] );
		}

		if ( version_compare( sekhlo_premium_get_theme_version(), '3.1.0-alpha.1', '>=' ) ) {
			unset( $modules['Colors'] );
		}

		$deprecated_modules = apply_filters(
			'sekhlo_premium_deprecated_modules',
			array(
				'Page Header',
				'Hooks',
				'Sections',
			)
		);

		foreach ( $deprecated_modules as $deprecated_module ) {
			if ( isset( $modules[ $deprecated_module ] ) ) {
				$modules[ $deprecated_module ]['deprecated'] = true;
			}
		}

		ksort( $modules );

		return $modules;
	}

	/**
	 * Get modules that can have their settings exported and imported.
	 */
	public static function get_exportable_modules() {
		$modules = array(
			'Core' => array(
				'settings' => 'sekhlo_settings',
				'title' => __( 'Core', 'sekhlopress-connect' ),
				'isActive' => true,
			),
		);

		foreach ( self::get_modules() as $key => $data ) {
			if ( ! empty( $data['exportable'] ) && $data['isActive'] ) {
				$modules[ $key ] = $data;
			}
		}

		return $modules;
	}

	/**
	 * Get options using theme_mods.
	 */
	public static function get_theme_mods() {
		$theme_mods = array(
			'font_body_variants',
			'font_body_category',
			'font_site_title_variants',
			'font_site_title_category',
			'font_site_tagline_variants',
			'font_site_tagline_category',
			'font_navigation_variants',
			'font_navigation_category',
			'font_secondary_navigation_variants',
			'font_secondary_navigation_category',
			'font_buttons_variants',
			'font_buttons_category',
			'font_heading_1_variants',
			'font_heading_1_category',
			'font_heading_2_variants',
			'font_heading_2_category',
			'font_heading_3_variants',
			'font_heading_3_category',
			'font_heading_4_variants',
			'font_heading_4_category',
			'font_heading_5_variants',
			'font_heading_5_category',
			'font_heading_6_variants',
			'font_heading_6_category',
			'font_widget_title_variants',
			'font_widget_title_category',
			'font_footer_variants',
			'font_footer_category',
			'sekhlo_copyright',
		);

		if ( function_exists( 'sekhlo_is_using_dynamic_typography' ) && sekhlo_is_using_dynamic_typography() ) {
			$theme_mods = array(
				'sekhlo_copyright',
			);
		}

		return $theme_mods;
	}

	/**
	 * Get our setting keys.
	 */
	public static function get_setting_keys() {
		return array(
			'sekhlo_settings',
			'sekhlo_background_settings',
			'sekhlo_blog_settings',
			'sekhlo_hooks',
			'sekhlo_page_header_settings',
			'sekhlo_secondary_nav_settings',
			'sekhlo_spacing_settings',
			'sekhlo_menu_plus_settings',
			'sekhlo_woocommerce_settings',
		);
	}

	/**
	 * Returns safely the license key.
	 */
	public static function get_license_key() {
		$license_key = get_option( 'gen_premium_license_key', '' );

		if ( $license_key && strlen( $license_key ) > 4 ) {
			$hidden_length = strlen( $license_key ) - 4;
			$safe_part = substr( $license_key, -4 );
			$hidden_part = implode( '', array_fill( 0, $hidden_length, '*' ) );

			return $hidden_part . $safe_part;
		}

		return $license_key;
	}

	/**
	 * Add our scripts to the page.
	 */
	public function enqueue_scripts() {
		if ( ! class_exists( 'SekhloPress_Dashboard' ) ) {
			return;
		}

		$dashboard_pages = SekhloPress_Dashboard::get_pages();
		$current_screen = get_current_screen();


		if ( in_array( $current_screen->id, $dashboard_pages ) ) {
			$packages_info = sekhlo_premium_get_enqueue_assets( 'packages' );
			wp_enqueue_style(
				'sekhlopress-pro-packages',
				GP_PREMIUM_DIR_URL . 'dist/packages.css',
				array(),
				$packages_info['version']
			);

			wp_enqueue_style(
				'sekhlo-pro-dashboard',
				GP_PREMIUM_DIR_URL . 'dist/style-dashboard.css',
				array( 'wp-components' ),
				SEKHLO_CONNECT_ADDON_VERSION
			);

			if ( 'appearance_page_sekhlo-options' === $current_screen->id ) {
				wp_enqueue_script(
					'sekhlo-pro-dashboard',
					GP_PREMIUM_DIR_URL . 'dist/dashboard.js',
					array(),
					SEKHLO_CONNECT_ADDON_VERSION,
					true
				);

				wp_set_script_translations( 'sekhlo-pro-dashboard', 'sekhlopress-connect', GP_PREMIUM_DIR_PATH . 'langs' );

				wp_localize_script(
					'sekhlo-pro-dashboard',
					'generateProDashboard',
					array(
						'modules' => self::get_modules(),
						'exportableModules' => self::get_exportable_modules(),
						'fontLibraryUrl' => admin_url( 'themes.php?page=sekhlopress-font-library' ),
						'siteLibraryUrl' => admin_url( 'themes.php?page=sekhlopress-library' ),
						'elementsUrl' => admin_url( 'edit.php?post_type=gp_elements' ),
						'hasWooCommerce' => class_exists( 'WooCommerce' ),
						'licenseKey' => self::get_license_key(),
						'licenseKeyStatus' => get_option( 'gen_premium_license_key_status', 'deactivated' ),
						'betaTester' => get_option( 'gp_premium_beta_testing', false ),
					)
				);
			}
		}
	}

	/**
	 * Enable beta testing if our option is set.
	 *
	 * @since 2.1.0
	 * @param boolean $value Whether beta testing is on or not.
	 */
	public function set_beta_tester( $value ) {
		if ( get_option( 'gp_premium_beta_testing', false ) ) {
			return true;
		}

		return $value;
	}

	/**
	 * Add the container for our start customizing app.
	 */
	public function module_list() {
		echo '<div id="sekhlopress-module-list"></div>';
	}

	/**
	 * Add the container for our start customizing app.
	 */
	public function license_key() {
		echo '<div id="sekhlopress-license-key"></div>';
	}

	/**
	 * Add the container for our start customizing app.
	 */
	public function import_export() {
		echo '<div id="sekhlopress-import-export-pro"></div>';
	}

	/**
	 * Add the container for our reset app.
	 */
	public function reset() {
		echo '<div id="sekhlopress-reset-pro"></div>';
	}
}

SekhloPress_Pro_Dashboard::get_instance();
