<?php
/**
 * This file handles the Disable Elements functionality.
 *
 * @package SekhloPress Connect Addon
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // No direct access, please.
}

define( 'sekhlo_DE_LAYOUT_META_BOX', true );

if ( ! function_exists( 'sekhlo_disable_elements' ) ) {
	/**
	 * Remove the default disable_elements.
	 *
	 * @since 0.1
	 */
	function sekhlo_disable_elements() {
		// Don't run the function unless we're on a page it applies to.
		if ( ! is_singular() ) {
			return '';
		}

		global $post;

		// Prevent PHP notices.
		if ( isset( $post ) ) {
			$disable_header = get_post_meta( $post->ID, '_sekhlo-disable-header', true );
			$disable_nav = get_post_meta( $post->ID, '_sekhlo-disable-nav', true );
			$disable_secondary_nav = get_post_meta( $post->ID, '_sekhlo-disable-secondary-nav', true );
			$disable_post_image = get_post_meta( $post->ID, '_sekhlo-disable-post-image', true );
			$disable_headline = get_post_meta( $post->ID, '_sekhlo-disable-headline', true );
			$disable_footer = get_post_meta( $post->ID, '_sekhlo-disable-footer', true );
		}

		$return = '';

		if ( ! empty( $disable_header ) && false !== $disable_header ) {
			$return = '.site-header {display:none}';
		}

		if ( ! empty( $disable_nav ) && false !== $disable_nav ) {
			$return .= '#site-navigation,.navigation-clone, #mobile-header {display:none !important}';
		}

		if ( ! empty( $disable_secondary_nav ) && false !== $disable_secondary_nav ) {
			$return .= '#secondary-navigation {display:none}';
		}

		if ( ! empty( $disable_post_image ) && false !== $disable_post_image ) {
			$return .= '.sekhlo-page-header, .page-header-image, .page-header-image-single {display:none}';
		}

		$need_css_removal = true;

		if ( defined( 'sekhlo_VERSION' ) && version_compare( sekhlo_VERSION, '3.0.0-alpha.1', '>=' ) ) {
			$need_css_removal = false;
		}

		if ( $need_css_removal && ! empty( $disable_headline ) && false !== $disable_headline && ! is_single() ) {
			$return .= '.entry-header {display:none} .page-content, .entry-content, .entry-summary {margin-top:0}';
		}

		if ( ! empty( $disable_footer ) && false !== $disable_footer ) {
			$return .= '.site-footer {display:none}';
		}

		return $return;
	}
}

if ( ! function_exists( 'sekhlo_de_scripts' ) ) {
	add_action( 'wp_enqueue_scripts', 'sekhlo_de_scripts', 50 );
	/**
	 * Enqueue scripts and styles
	 */
	function sekhlo_de_scripts() {
		wp_add_inline_style( 'sekhlo-style', sekhlo_disable_elements() );
	}
}

if ( ! function_exists( 'sekhlo_add_de_meta_box' ) ) {
	add_action( 'add_meta_boxes', 'sekhlo_add_de_meta_box', 50 );
	/**
	 * Generate the layout metabox.
	 *
	 * @since 0.1
	 */
	function sekhlo_add_de_meta_box() {
		// Set user role - make filterable.
		$allowed = apply_filters( 'sekhlo_metabox_capability', 'edit_theme_options' );

		// If not an administrator, don't show the metabox.
		if ( ! current_user_can( $allowed ) ) {
			return;
		}

		if ( defined( 'sekhlo_LAYOUT_META_BOX' ) ) {
			return;
		}

		$args = array( 'public' => true );
		$post_types = get_post_types( $args );
		foreach ( $post_types as $type ) {
			if ( 'attachment' !== $type ) {
				add_meta_box(
					'sekhlo_de_meta_box',
					__( 'Disable Elements', 'sekhlopress-connect' ),
					'sekhlo_show_de_meta_box',
					$type,
					'side',
					'default'
				);
			}
		}
	}
}

if ( ! function_exists( 'sekhlo_show_de_meta_box' ) ) {
	/**
	 * Outputs the content of the metabox.
	 *
	 * @param object $post The post object.
	 */
	function sekhlo_show_de_meta_box( $post ) {
		wp_nonce_field( basename( __FILE__ ), 'sekhlo_de_nonce' );
		$stored_meta = get_post_meta( $post->ID );
		$stored_meta['_sekhlo-disable-header'][0] = ( isset( $stored_meta['_sekhlo-disable-header'][0] ) ) ? $stored_meta['_sekhlo-disable-header'][0] : '';
		$stored_meta['_sekhlo-disable-nav'][0] = ( isset( $stored_meta['_sekhlo-disable-nav'][0] ) ) ? $stored_meta['_sekhlo-disable-nav'][0] : '';
		$stored_meta['_sekhlo-disable-secondary-nav'][0] = ( isset( $stored_meta['_sekhlo-disable-secondary-nav'][0] ) ) ? $stored_meta['_sekhlo-disable-secondary-nav'][0] : '';
		$stored_meta['_sekhlo-disable-post-image'][0] = ( isset( $stored_meta['_sekhlo-disable-post-image'][0] ) ) ? $stored_meta['_sekhlo-disable-post-image'][0] : '';
		$stored_meta['_sekhlo-disable-headline'][0] = ( isset( $stored_meta['_sekhlo-disable-headline'][0] ) ) ? $stored_meta['_sekhlo-disable-headline'][0] : '';
		$stored_meta['_sekhlo-disable-footer'][0] = ( isset( $stored_meta['_sekhlo-disable-footer'][0] ) ) ? $stored_meta['_sekhlo-disable-footer'][0] : '';
		$stored_meta['_sekhlo-disable-top-bar'][0] = ( isset( $stored_meta['_sekhlo-disable-top-bar'][0] ) ) ? $stored_meta['_sekhlo-disable-top-bar'][0] : '';
		?>

		<p>
			<div class="sekhlo_disable_elements">
				<?php if ( function_exists( 'sekhlo_top_bar' ) ) : ?>
					<label for="meta-sekhlo-disable-top-bar" style="display:block;margin-bottom:3px;" title="<?php _e( 'Top Bar', 'sekhlopress-connect' ); ?>">
						<input type="checkbox" name="_sekhlo-disable-top-bar" id="meta-sekhlo-disable-top-bar" value="true" <?php checked( $stored_meta['_sekhlo-disable-top-bar'][0], 'true' ); ?>>
						<?php _e( 'Top Bar', 'sekhlopress-connect' ); ?>
					</label>
				<?php endif; ?>

				<label for="meta-sekhlo-disable-header" style="display:block;margin-bottom:3px;" title="<?php _e( 'Header', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-header" id="meta-sekhlo-disable-header" value="true" <?php checked( $stored_meta['_sekhlo-disable-header'][0], 'true' ); ?>>
					<?php _e( 'Header', 'sekhlopress-connect' ); ?>
				</label>

				<label for="meta-sekhlo-disable-nav" style="display:block;margin-bottom:3px;" title="<?php _e( 'Primary Navigation', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-nav" id="meta-sekhlo-disable-nav" value="true" <?php checked( $stored_meta['_sekhlo-disable-nav'][0], 'true' ); ?>>
					<?php _e( 'Primary Navigation', 'sekhlopress-connect' ); ?>
				</label>

				<?php if ( function_exists( 'sekhlo_secondary_nav_setup' ) ) : ?>
					<label for="meta-sekhlo-disable-secondary-nav" style="display:block;margin-bottom:3px;" title="<?php _e( 'Secondary Navigation', 'sekhlopress-connect' ); ?>">
						<input type="checkbox" name="_sekhlo-disable-secondary-nav" id="meta-sekhlo-disable-secondary-nav" value="true" <?php checked( $stored_meta['_sekhlo-disable-secondary-nav'][0], 'true' ); ?>>
						<?php _e( 'Secondary Navigation', 'sekhlopress-connect' ); ?>
					</label>
				<?php endif; ?>

				<label for="meta-sekhlo-disable-post-image" style="display:block;margin-bottom:3px;" title="<?php _e( 'Featured Image', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-post-image" id="meta-sekhlo-disable-post-image" value="true" <?php checked( $stored_meta['_sekhlo-disable-post-image'][0], 'true' ); ?>>
					<?php _e( 'Featured Image', 'sekhlopress-connect' ); ?>
				</label>

				<label for="meta-sekhlo-disable-headline" style="display:block;margin-bottom:3px;" title="<?php _e( 'Content Title', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-headline" id="meta-sekhlo-disable-headline" value="true" <?php checked( $stored_meta['_sekhlo-disable-headline'][0], 'true' ); ?>>
					<?php _e( 'Content Title', 'sekhlopress-connect' ); ?>
				</label>

				<label for="meta-sekhlo-disable-footer" style="display:block;margin-bottom:3px;" title="<?php _e( 'Footer', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-footer" id="meta-sekhlo-disable-footer" value="true" <?php checked( $stored_meta['_sekhlo-disable-footer'][0], 'true' ); ?>>
					<?php _e( 'Footer', 'sekhlopress-connect' ); ?>
				</label>
			</div>
		</p>

		<?php
	}
}

if ( ! function_exists( 'sekhlo_save_de_meta' ) ) {
	add_action( 'save_post', 'sekhlo_save_de_meta' );
	/**
	 * Save our options.
	 *
	 * @param int $post_id The post ID.
	 */
	function sekhlo_save_de_meta( $post_id ) {

		if ( defined( 'sekhlo_LAYOUT_META_BOX' ) ) {
			return;
		}

		// Checks save status.
		$is_autosave = wp_is_post_autosave( $post_id );
		$is_revision = wp_is_post_revision( $post_id );
		$is_valid_nonce = ( isset( $_POST['sekhlo_de_nonce'] ) && wp_verify_nonce( $_POST['sekhlo_de_nonce'], basename( __FILE__ ) ) ) ? true : false;

		// Exits script depending on save status.
		if ( $is_autosave || $is_revision || ! $is_valid_nonce ) {
			return;
		}

		// Check that the logged in user has permission to edit this post.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return $post_id;
		}

		$options = array(
			'_sekhlo-disable-top-bar',
			'_sekhlo-disable-header',
			'_sekhlo-disable-nav',
			'_sekhlo-disable-secondary-nav',
			'_sekhlo-disable-headline',
			'_sekhlo-disable-footer',
			'_sekhlo-disable-post-image',
		);

		foreach ( $options as $key ) {
			$value = isset( $_POST[ $key ] )
				? sanitize_text_field( wp_unslash( $_POST[ $key ] ) )
				: '';

			if ( $value ) {
				update_post_meta( $post_id, $key, $value );
			} else {
				delete_post_meta( $post_id, $key );
			}
		}
	}
}

if ( ! function_exists( 'sekhlo_disable_elements_setup' ) ) {
	add_action( 'wp', 'sekhlo_disable_elements_setup', 50 );
	/**
	 * Disable the things.
	 */
	function sekhlo_disable_elements_setup() {
		// Don't run the function unless we're on a page it applies to.
		if ( ! is_singular() ) {
			return;
		}

		// Get the current post.
		global $post;

		// Grab our values.
		if ( isset( $post ) ) {
			$disable_top_bar = get_post_meta( $post->ID, '_sekhlo-disable-top-bar', true );
			$disable_header = get_post_meta( $post->ID, '_sekhlo-disable-header', true );
			$disable_mobile_header = get_post_meta( $post->ID, '_sekhlo-disable-mobile-header', true );
			$disable_nav = get_post_meta( $post->ID, '_sekhlo-disable-nav', true );
			$disable_headline = get_post_meta( $post->ID, '_sekhlo-disable-headline', true );
			$disable_footer = get_post_meta( $post->ID, '_sekhlo-disable-footer', true );
		}

		// Remove the top bar.
		if ( ! empty( $disable_top_bar ) && false !== $disable_top_bar && function_exists( 'sekhlo_top_bar' ) ) {
			remove_action( 'sekhlo_before_header', 'sekhlo_top_bar', 5 );
			remove_action( 'sekhlo_inside_secondary_navigation', 'sekhlo_secondary_nav_top_bar_widget', 5 );
		}

		// Remove the header.
		if ( ! empty( $disable_header ) && false !== $disable_header && function_exists( 'sekhlo_construct_header' ) ) {
			remove_action( 'sekhlo_header', 'sekhlo_construct_header' );
		}

		// Remove the mobile header.
		if ( ! empty( $disable_mobile_header ) && false !== $disable_mobile_header && function_exists( 'sekhlo_menu_plus_mobile_header' ) ) {
			remove_action( 'sekhlo_after_header', 'sekhlo_menu_plus_mobile_header', 5 );
		}

		// Remove the navigation.
		if ( ! empty( $disable_nav ) && false !== $disable_nav && function_exists( 'sekhlo_get_navigation_location' ) ) {
			add_filter( 'sekhlo_navigation_location', '__return_false', 20 );
			add_filter( 'sekhlo_disable_mobile_header_menu', '__return_true' );
		}

		// Remove the title.
		if ( ! empty( $disable_headline ) && false !== $disable_headline && function_exists( 'sekhlo_show_title' ) ) {
			add_filter( 'sekhlo_show_title', '__return_false' );
		}

		// Remove the footer.
		if ( ! empty( $disable_footer ) && false !== $disable_footer ) {
			if ( function_exists( 'sekhlo_construct_footer_widgets' ) ) {
				remove_action( 'sekhlo_footer', 'sekhlo_construct_footer_widgets', 5 );
			}

			if ( function_exists( 'sekhlo_construct_footer' ) ) {
				remove_action( 'sekhlo_footer', 'sekhlo_construct_footer' );
			}
		}
	}
}

add_action( 'sekhlo_layout_disable_elements_section', 'sekhlo_premium_disable_elements_options' );
/**
 * Add the meta box options to the Layout meta box in the new GP
 *
 * @since 1.4
 * @param array $stored_meta Existing meta data.
 */
function sekhlo_premium_disable_elements_options( $stored_meta ) {
	$stored_meta['_sekhlo-disable-header'][0] = ( isset( $stored_meta['_sekhlo-disable-header'][0] ) ) ? $stored_meta['_sekhlo-disable-header'][0] : '';
	$stored_meta['_sekhlo-disable-mobile-header'][0] = ( isset( $stored_meta['_sekhlo-disable-mobile-header'][0] ) ) ? $stored_meta['_sekhlo-disable-mobile-header'][0] : '';
	$stored_meta['_sekhlo-disable-nav'][0] = ( isset( $stored_meta['_sekhlo-disable-nav'][0] ) ) ? $stored_meta['_sekhlo-disable-nav'][0] : '';
	$stored_meta['_sekhlo-disable-secondary-nav'][0] = ( isset( $stored_meta['_sekhlo-disable-secondary-nav'][0] ) ) ? $stored_meta['_sekhlo-disable-secondary-nav'][0] : '';
	$stored_meta['_sekhlo-disable-post-image'][0] = ( isset( $stored_meta['_sekhlo-disable-post-image'][0] ) ) ? $stored_meta['_sekhlo-disable-post-image'][0] : '';
	$stored_meta['_sekhlo-disable-headline'][0] = ( isset( $stored_meta['_sekhlo-disable-headline'][0] ) ) ? $stored_meta['_sekhlo-disable-headline'][0] : '';
	$stored_meta['_sekhlo-disable-footer'][0] = ( isset( $stored_meta['_sekhlo-disable-footer'][0] ) ) ? $stored_meta['_sekhlo-disable-footer'][0] : '';
	$stored_meta['_sekhlo-disable-top-bar'][0] = ( isset( $stored_meta['_sekhlo-disable-top-bar'][0] ) ) ? $stored_meta['_sekhlo-disable-top-bar'][0] : '';
	?>
	<div class="sekhlo_disable_elements">
		<?php if ( function_exists( 'sekhlo_top_bar' ) ) : ?>
			<label for="meta-sekhlo-disable-top-bar" style="display:block;margin-bottom:3px;" title="<?php _e( 'Top Bar', 'sekhlopress-connect' ); ?>">
				<input type="checkbox" name="_sekhlo-disable-top-bar" id="meta-sekhlo-disable-top-bar" value="true" <?php checked( $stored_meta['_sekhlo-disable-top-bar'][0], 'true' ); ?>>
				<?php _e( 'Top Bar', 'sekhlopress-connect' ); ?>
			</label>
		<?php endif; ?>

		<label for="meta-sekhlo-disable-header" style="display:block;margin-bottom:3px;" title="<?php _e( 'Header', 'sekhlopress-connect' ); ?>">
			<input type="checkbox" name="_sekhlo-disable-header" id="meta-sekhlo-disable-header" value="true" <?php checked( $stored_meta['_sekhlo-disable-header'][0], 'true' ); ?>>
			<?php _e( 'Header', 'sekhlopress-connect' ); ?>
		</label>

		<?php
		if ( function_exists( 'sekhlo_menu_plus_get_defaults' ) ) :
			$menu_plus_settings = wp_parse_args(
				get_option( 'sekhlo_menu_plus_settings', array() ),
				sekhlo_menu_plus_get_defaults()
			);

			if ( 'enable' === $menu_plus_settings['mobile_header'] ) :
				?>
				<label for="meta-sekhlo-disable-mobile-header" style="display:block;margin-bottom:3px;" title="<?php esc_attr_e( 'Mobile Header', 'sekhlopress-connect' ); ?>">
					<input type="checkbox" name="_sekhlo-disable-mobile-header" id="meta-sekhlo-disable-mobile-header" value="true" <?php checked( $stored_meta['_sekhlo-disable-mobile-header'][0], 'true' ); ?>>
					<?php esc_html_e( 'Mobile Header', 'sekhlopress-connect' ); ?>
				</label>
				<?php
			endif;
		endif;
		?>

		<label for="meta-sekhlo-disable-nav" style="display:block;margin-bottom:3px;" title="<?php _e( 'Primary Navigation', 'sekhlopress-connect' ); ?>">
			<input type="checkbox" name="_sekhlo-disable-nav" id="meta-sekhlo-disable-nav" value="true" <?php checked( $stored_meta['_sekhlo-disable-nav'][0], 'true' ); ?>>
			<?php _e( 'Primary Navigation', 'sekhlopress-connect' ); ?>
		</label>

		<?php if ( function_exists( 'sekhlo_secondary_nav_setup' ) ) : ?>
			<label for="meta-sekhlo-disable-secondary-nav" style="display:block;margin-bottom:3px;" title="<?php _e( 'Secondary Navigation', 'sekhlopress-connect' ); ?>">
				<input type="checkbox" name="_sekhlo-disable-secondary-nav" id="meta-sekhlo-disable-secondary-nav" value="true" <?php checked( $stored_meta['_sekhlo-disable-secondary-nav'][0], 'true' ); ?>>
				<?php _e( 'Secondary Navigation', 'sekhlopress-connect' ); ?>
			</label>
		<?php endif; ?>

		<label for="meta-sekhlo-disable-post-image" style="display:block;margin-bottom:3px;" title="<?php _e( 'Featured Image', 'sekhlopress-connect' ); ?>">
			<input type="checkbox" name="_sekhlo-disable-post-image" id="meta-sekhlo-disable-post-image" value="true" <?php checked( $stored_meta['_sekhlo-disable-post-image'][0], 'true' ); ?>>
			<?php _e( 'Featured Image', 'sekhlopress-connect' ); ?>
		</label>

		<label for="meta-sekhlo-disable-headline" style="display:block;margin-bottom:3px;" title="<?php _e( 'Content Title', 'sekhlopress-connect' ); ?>">
			<input type="checkbox" name="_sekhlo-disable-headline" id="meta-sekhlo-disable-headline" value="true" <?php checked( $stored_meta['_sekhlo-disable-headline'][0], 'true' ); ?>>
			<?php _e( 'Content Title', 'sekhlopress-connect' ); ?>
		</label>

		<label for="meta-sekhlo-disable-footer" style="display:block;margin-bottom:3px;" title="<?php _e( 'Footer', 'sekhlopress-connect' ); ?>">
			<input type="checkbox" name="_sekhlo-disable-footer" id="meta-sekhlo-disable-footer" value="true" <?php checked( $stored_meta['_sekhlo-disable-footer'][0], 'true' ); ?>>
			<?php _e( 'Footer', 'sekhlopress-connect' ); ?>
		</label>
	</div>
	<?php
}

add_action( 'sekhlo_layout_meta_box_save', 'sekhlo_premium_save_disable_elements_meta' );
/**
 * Save the Disable Elements meta box values
 *
 * @since 1.4
 * @param int $post_id The post ID.
 */
function sekhlo_premium_save_disable_elements_meta( $post_id ) {
	$options = array(
		'_sekhlo-disable-top-bar',
		'_sekhlo-disable-header',
		'_sekhlo-disable-mobile-header',
		'_sekhlo-disable-nav',
		'_sekhlo-disable-secondary-nav',
		'_sekhlo-disable-headline',
		'_sekhlo-disable-footer',
		'_sekhlo-disable-post-image',
	);

	foreach ( $options as $key ) {
		$value = isset( $_POST[ $key ] ) // phpcs:ignore -- Nonce exists within `sekhlo_layout_meta_box_save` hook.
			? sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) // phpcs:ignore -- Nonce exists within `sekhlo_layout_meta_box_save` hook.
			: '';

		if ( $value ) {
			update_post_meta( $post_id, $key, $value );
		} else {
			delete_post_meta( $post_id, $key );
		}
	}
}

add_filter( 'body_class', 'sekhlo_disable_elements_body_classes', 20 );
/**
 * Remove body classes if certain elements are disabled.
 *
 * @since 2.1.0
 * @param array $classes Our body classes.
 */
function sekhlo_disable_elements_body_classes( $classes ) {
	if ( is_singular() ) {
		$disable_featured_image = get_post_meta( get_the_ID(), '_sekhlo-disable-post-image', true );
		$classes = sekhlo_premium_remove_featured_image_class( $classes, $disable_featured_image );
	}

	return $classes;
}
