<?php
/**
 * Plugin Name: Sekhlo Schema Code
 * Plugin URI: https://www.mubashirhassan.com/sekhlo-schema-code.html
 * Description: Advanced Schema Markup Manager optimized for SEO, AI Search, and Voice Search. Features AI-powered prompt generators, templates, validation, and multiple schema support.
 * Version: 1.0.0
 * Author: Mubashir Hassan
 * Author URI: https://www.mubashirhassan.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: sekhlo-schema-code
 * Requires at least: 5.0
 * Requires PHP: 7.2
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define Constants
define('SEKHLO_SCHEMA_PATH', plugin_dir_path(__FILE__));
define('SEKHLO_SCHEMA_URL', plugin_dir_url(__FILE__));

// Include Core Classes
require_once SEKHLO_SCHEMA_PATH . 'includes/class-schema-manager.php';
require_once SEKHLO_SCHEMA_PATH . 'includes/class-schema-templates.php';
require_once SEKHLO_SCHEMA_PATH . 'includes/class-settings-page.php';

class Sekhlo_Schema_Plugin {

    private $schema_manager;

    public function __construct() {
        $this->schema_manager = new Sekhlo_Schema_Manager();

        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('wp_head', array($this, 'output_schema_json_ld'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'sekhlo_add_plugin_action_links'));
        add_action('wp_head', array($this, 'output_seo_meta_tags'));
        add_filter('pre_get_document_title', array($this, 'filter_seo_title'), 20);
    }

    /**
     * Enqueue Admin Assets
     */
    public function enqueue_admin_assets($hook) {
        if ('post.php' != $hook && 'post-new.php' != $hook) {
            return;
        }

        wp_enqueue_style('sekhlo-admin-styles', SEKHLO_SCHEMA_URL . 'assets/css/admin-styles.css', array(), '2.3.0');
        wp_enqueue_script('sekhlo-admin-scripts', SEKHLO_SCHEMA_URL . 'assets/js/admin-scripts.js', array('jquery', 'jquery-ui-sortable'), '2.3.0', true);

        // Localize data for JS
        global $post;
        if ($post) {
            // Get schemas for this post
            $schemas = $this->schema_manager->get_schemas($post->ID);
            
            wp_localize_script('sekhlo-admin-scripts', 'sekhloData', array(
                'nonce'     => wp_create_nonce('sekhlo_schema_nonce'),
                'postTitle' => get_the_title($post->ID),
                'postUrl'   => get_permalink($post->ID),
                'siteTitle' => get_bloginfo('name'),
                'siteUrl'   => home_url('/'),
                'featuredImageUrl' => get_the_post_thumbnail_url($post->ID, 'medium'),
                'postContent' => wp_strip_all_tags($post->post_content),
                'keyphrases' => (array)get_post_meta($post->ID, 'sekhlo_seo_keyphrases', true),
                'metaTitle' => get_post_meta($post->ID, 'sekhlo_seo_title', true),
                'metaDescription' => get_post_meta($post->ID, 'sekhlo_seo_description', true),
                // Pass templates to JS
                'templates' => Sekhlo_Schema_Templates::get_templates(),
                // Pass schemas directly in localized data for additional reliability
                'schemas' => $schemas
            ));
            
            // Also pass schemas via inline script for backward compatibility
            wp_add_inline_script('sekhlo-admin-scripts', 'window.schemas = ' . wp_json_encode($schemas) . '; window.sekhloTemplates = ' . wp_json_encode(Sekhlo_Schema_Templates::get_templates()) . ';', 'before');
        }
    }

    /**
     * Add Meta Boxes
     */
    public function add_meta_boxes() {
        $screens = $this->schema_manager->get_supported_post_types();
        foreach ($screens as $screen) {
            add_meta_box(
                'sekhlo_schema_meta_box',
                'Sekhlo Schema Manager (AI & SEO Optimized)',
                array($this, 'render_meta_box'),
                $screen,
                'normal',
                'high'
            );
        }
    }

    /**
     * Render Meta Box
     */
    public function render_meta_box($post) {
        wp_nonce_field('sekhlo_save_schemas', 'sekhlo_schema_nonce');
        
        $schemas = $this->schema_manager->get_schemas($post->ID);
        $templates = Sekhlo_Schema_Templates::get_templates();

        // Include the template
        include SEKHLO_SCHEMA_PATH . 'templates/meta-box.php';
    }

    /**
     * Filter SEO Title
     */
    public function filter_seo_title($title) {
        if (is_singular()) {
            global $post;
            $custom_title = get_post_meta($post->ID, 'sekhlo_seo_title', true);
            if (!empty($custom_title)) {
                return $custom_title;
            }
        }
        return $title;
    }

    /**
     * Output SEO Meta Tags
     */
    public function output_seo_meta_tags() {
        if (is_singular()) {
            global $post;
            $desc = get_post_meta($post->ID, 'sekhlo_seo_description', true);
            $img = get_the_post_thumbnail_url($post->ID, 'large');
            $title = get_post_meta($post->ID, 'sekhlo_seo_title', true) ?: get_the_title($post->ID);

            if (!empty($desc)) {
                echo '<meta name="description" content="' . esc_attr($desc) . '">' . "\n";
            }

            // OpenGraph
            echo '<meta property="og:title" content="' . esc_attr($title) . '">' . "\n";
            if (!empty($desc)) {
                echo '<meta property="og:description" content="' . esc_attr($desc) . '">' . "\n";
            }
            if ($img) {
                echo '<meta property="og:image" content="' . esc_url($img) . '">' . "\n";
            }
            echo '<meta property="og:type" content="article">' . "\n";
            echo '<meta property="og:url" content="' . esc_url(get_permalink($post->ID)) . '">' . "\n";
        }
    }

    /**
     * Add Settings Link to Plugin List
     */
    public function sekhlo_add_plugin_action_links($links) {
        $settings_link = '<a href="options-general.php?page=sekhlo-schema-settings">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Output Schema in Head
     */
    public function output_schema_json_ld() {
        if (is_singular() || is_front_page()) {
            $post_id = get_the_ID();
            if (is_front_page() && !is_singular()) {
                $post_id = get_option('page_on_front');
            }
            
            if (!$post_id) return;
            
            $schemas = $this->schema_manager->get_schemas($post_id);
            
            if (!empty($schemas)) {
                foreach ($schemas as $schema) {
                    if (isset($schema['enabled']) && $schema['enabled'] && !empty($schema['code'])) {
                        // Minify JSON if desired, but pretty print is fine for debug.
                        echo '<!-- Sekhlo Schema: ' . esc_html($schema['name']) . ' -->' . "\n";
                        echo '<script type="application/ld+json">' . "\n";
                        echo $schema['code'] . "\n"; // Already sanitized or trusted admin input
                        echo '</script>' . "\n";
                    }
                }
            }
        }
    }
}

// Initialize Plugin
new Sekhlo_Schema_Plugin();