<?php
/**
 * Settings Page Class
 * Handles admin settings for llms.txt, robots.txt, and .htaccess
 */

if (!defined('ABSPATH')) {
    exit;
}

class Sekhlo_Settings_Page {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_settings_styles'));
        add_action('init', array($this, 'handle_virtual_files'));
        add_action('wp_ajax_sekhlo_save_global_settings', array($this, 'ajax_save_settings'));
    }

    /**
     * Enqueue Settings Page Styles
     */
    public function enqueue_settings_styles($hook) {
        // Only load on our settings page
        if ('settings_page_sekhlo-schema-settings' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'sekhlo-settings-page',
            SEKHLO_SCHEMA_URL . 'assets/css/settings-page.css',
            array(),
            '1.0.0'
        );
    }

    /**
     * Handle Virtual Files (llms.txt and sitemap.xml)
     */
    public function handle_virtual_files() {
        $this->generate_llms_txt();
        $this->generate_sitemap_xml();
    }

    /**
     * AJAX Save Settings
     */
    public function ajax_save_settings() {
        check_ajax_referer('sekhlo_schema_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        // Save llms.txt settings
        update_option('sekhlo_llms_enabled', intval($_POST['llms_enabled']));
        update_option('sekhlo_llms_content', wp_unslash($_POST['llms_content']));

        // Save sitemap settings
        if (isset($_POST['sitemap_enabled'])) {
            update_option('sekhlo_sitemap_enabled', intval($_POST['sitemap_enabled']));
            update_option('sekhlo_sitemap_include_posts', intval($_POST['sitemap_include_posts']));
            update_option('sekhlo_sitemap_include_pages', intval($_POST['sitemap_include_pages']));
            update_option('sekhlo_sitemap_include_categories', intval($_POST['sitemap_include_categories']));
        }

        // Save robots.txt
        if (isset($_POST['robots_content'])) {
            $this->save_robots_txt(wp_unslash($_POST['robots_content']));
        }

        // Save .htaccess (with backup handled by helper)
        if (isset($_POST['htaccess_content'])) {
            $this->save_htaccess(wp_unslash($_POST['htaccess_content']));
        }

        wp_send_json_success('Settings saved');
    }

    /**
     * Add settings page to admin menu
     */
    public function add_settings_page() {
        add_options_page(
            'Sekhlo Schema Settings',
            'Sekhlo Schema',
            'manage_options',
            'sekhlo-schema-settings',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('sekhlo_settings_group', 'sekhlo_llms_enabled', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_llms_content', array(
            'sanitize_callback' => 'wp_kses_post'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_sitemap_enabled', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_sitemap_include_posts', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_sitemap_include_pages', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_sitemap_include_categories', array(
            'sanitize_callback' => 'absint'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_robots_content', array(
            'sanitize_callback' => 'sanitize_textarea_field'
        ));
        register_setting('sekhlo_settings_group', 'sekhlo_htaccess_content', array(
            'sanitize_callback' => 'sanitize_textarea_field'
        ));
    }

    /**
     * Generate llms.txt file
     */
    public function generate_llms_txt() {
        if (!get_option('sekhlo_llms_enabled')) {
            return;
        }

        $request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        if ($request_uri === '/llms.txt' || $request_uri === '/llms.txt/') {
            $content = get_option('sekhlo_llms_content', $this->get_default_llms_content());
            header('Content-Type: text/plain; charset=utf-8');
            echo $content;
            exit;
        }
    }

    /**
     * Generate sitemap.xml file
     */
    public function generate_sitemap_xml() {
        if (!get_option('sekhlo_sitemap_enabled')) {
            return;
        }

        $request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        if ($request_uri === '/sitemap.xml' || $request_uri === '/sitemap.xml/') {
            header('Content-Type: text/xml; charset=utf-8');
            
            $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
            $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

            // Home page
            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . esc_url(home_url('/')) . '</loc>' . "\n";
            $xml .= '    <changefreq>daily</changefreq>' . "\n";
            $xml .= '    <priority>1.0</priority>' . "\n";
            $xml .= '  </url>' . "\n";

            // Pages and Posts
            $post_types = array();
            if (get_option('sekhlo_sitemap_include_posts', 1)) $post_types[] = 'post';
            if (get_option('sekhlo_sitemap_include_pages', 1)) $post_types[] = 'page';

            if (!empty($post_types)) {
                $query = new WP_Query(array(
                    'post_type'      => $post_types,
                    'post_status'    => 'publish',
                    'posts_per_page' => -1,
                    'orderby'        => 'date',
                    'order'          => 'DESC'
                ));

                if ($query->have_posts()) {
                    while ($query->have_posts()) {
                        $query->the_post();
                        $xml .= '  <url>' . "\n";
                        $xml .= '    <loc>' . esc_url(get_permalink()) . '</loc>' . "\n";
                        $xml .= '    <lastmod>' . get_the_modified_date('c') . '</lastmod>' . "\n";
                        $xml .= '    <changefreq>weekly</changefreq>' . "\n";
                        $xml .= '    <priority>0.8</priority>' . "\n";
                        $xml .= '  </url>' . "\n";
                    }
                    wp_reset_postdata();
                }
            }

            // Categories
            if (get_option('sekhlo_sitemap_include_categories', 1)) {
                $categories = get_terms(array(
                    'taxonomy'   => 'category',
                    'hide_empty' => true,
                ));
                if (!empty($categories) && !is_wp_error($categories)) {
                    foreach ($categories as $category) {
                        $xml .= '  <url>' . "\n";
                        $xml .= '    <loc>' . esc_url(get_term_link($category)) . '</loc>' . "\n";
                        $xml .= '    <changefreq>weekly</changefreq>' . "\n";
                        $xml .= '    <priority>0.5</priority>' . "\n";
                        $xml .= '  </url>' . "\n";
                    }
                }
            }

            $xml .= '</urlset>';
            echo $xml;
            exit;
        }
    }

    /**
     * Get default llms.txt content
     */
    private function get_default_llms_content() {
        $site_name = get_bloginfo('name');
        $site_desc = get_bloginfo('description');
        $site_url = home_url('/');

        $content = "# {$site_name}\n";
        $content .= "> {$site_desc}\n\n";
        $content .= "## About\n";
        $content .= "This is the official website for {$site_name}.\n\n";
        $content .= "## Main Pages\n";

        // Get main pages
        $pages = get_pages(array('number' => 10, 'sort_column' => 'menu_order'));
        foreach ($pages as $page) {
            $content .= "- [{$page->post_title}](" . get_permalink($page->ID) . ")\n";
        }

        $content .= "\n## Recent Posts\n";

        // Get recent posts
        $posts = get_posts(array('numberposts' => 10));
        foreach ($posts as $post) {
            $content .= "- [{$post->post_title}](" . get_permalink($post->ID) . ")\n";
        }

        return $content;
    }

    /**
     * Get robots.txt content
     */
    private function get_robots_content() {
        $robots_file = ABSPATH . 'robots.txt';
        if (file_exists($robots_file)) {
            return file_get_contents($robots_file);
        }
        // Default WordPress robots.txt content
        return "User-agent: *\nDisallow: /wp-admin/\nAllow: /wp-admin/admin-ajax.php\n\nSitemap: " . home_url('/sitemap.xml');
    }

    /**
     * Get .htaccess content
     */
    private function get_htaccess_content() {
        $htaccess_file = ABSPATH . '.htaccess';
        if (file_exists($htaccess_file)) {
            return file_get_contents($htaccess_file);
        }
        return "# No .htaccess file found";
    }

    /**
     * Save robots.txt
     */
    private function save_robots_txt($content) {
        $robots_file = ABSPATH . 'robots.txt';
        return file_put_contents($robots_file, $content);
    }

    /**
     * Save .htaccess
     */
    private function save_htaccess($content) {
        $htaccess_file = ABSPATH . '.htaccess';
        // Create backup first
        if (file_exists($htaccess_file)) {
            copy($htaccess_file, ABSPATH . '.htaccess.backup');
        }
        return file_put_contents($htaccess_file, $content);
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        // Handle form submissions
        if (isset($_POST['sekhlo_save_settings']) && check_admin_referer('sekhlo_settings_nonce')) {
            update_option('sekhlo_llms_enabled', isset($_POST['sekhlo_llms_enabled']) ? 1 : 0);
            update_option('sekhlo_llms_content', wp_unslash($_POST['sekhlo_llms_content']));
            update_option('sekhlo_sitemap_enabled', isset($_POST['sekhlo_sitemap_enabled']) ? 1 : 0);
            update_option('sekhlo_sitemap_include_posts', isset($_POST['sekhlo_sitemap_include_posts']) ? 1 : 0);
            update_option('sekhlo_sitemap_include_pages', isset($_POST['sekhlo_sitemap_include_pages']) ? 1 : 0);
            update_option('sekhlo_sitemap_include_categories', isset($_POST['sekhlo_sitemap_include_categories']) ? 1 : 0);
            echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
        }

        if (isset($_POST['sekhlo_save_robots']) && check_admin_referer('sekhlo_settings_nonce')) {
            $this->save_robots_txt(wp_unslash($_POST['sekhlo_robots_content']));
            echo '<div class="notice notice-success"><p>robots.txt saved successfully!</p></div>';
        }

        if (isset($_POST['sekhlo_save_htaccess']) && check_admin_referer('sekhlo_settings_nonce')) {
            $this->save_htaccess(wp_unslash($_POST['sekhlo_htaccess_content']));
            echo '<div class="notice notice-success"><p>.htaccess saved successfully! A backup was created as .htaccess.backup</p></div>';
        }

        $llms_enabled = get_option('sekhlo_llms_enabled', 0);
        $sitemap_enabled = get_option('sekhlo_sitemap_enabled', 0);
        $sitemap_posts = get_option('sekhlo_sitemap_include_posts', 1);
        $sitemap_pages = get_option('sekhlo_sitemap_include_pages', 1);
        $sitemap_cats = get_option('sekhlo_sitemap_include_categories', 1);
        $llms_content = get_option('sekhlo_llms_content', $this->get_default_llms_content());
        $robots_content = $this->get_robots_content();
        $htaccess_content = $this->get_htaccess_content();
        ?>
        <div class="wrap">
            <h1>Sekhlo Schema Settings</h1>
            
            <form method="post">
                <?php wp_nonce_field('sekhlo_settings_nonce'); ?>

                <div class="sekhlo-settings-card">
                    <h2>
                        <span>🤖 llms.txt Manager <span class="sekhlo-badge ai">AI Optimization</span></span>
                        <?php if ($llms_enabled): ?>
                            <a href="<?php echo home_url('/llms.txt'); ?>" target="_blank" class="button button-small">View File ↗</a>
                        <?php endif; ?>
                    </h2>
                    
                    <p class="description">Generate a specialized file that helps AI assistants (ChatGPT, Claude, etc.) understand your website context and structure.</p>
                    
                    <div class="sekhlo-toggle-row">
                        <label>
                            <input type="checkbox" name="sekhlo_llms_enabled" value="1" <?php checked($llms_enabled, 1); ?>>
                            <strong>Enable llms.txt generation</strong>
                        </label>
                    </div>

                    <label class="description"><strong>Customize File Content (Markdown):</strong></label>
                    <textarea name="sekhlo_llms_content" class="sekhlo-code-editor"><?php echo esc_textarea($llms_content); ?></textarea>
                </div>

                <!-- XML Sitemap Section -->
                <div class="sekhlo-settings-card">
                    <h2>
                        <span>🗺️ XML Sitemap Generator <span class="sekhlo-badge ai">SEO</span></span>
                        <?php if ($sitemap_enabled): ?>
                            <a href="<?php echo home_url('/sitemap.xml'); ?>" target="_blank" class="button button-small">View Sitemap ↗</a>
                        <?php endif; ?>
                    </h2>
                    
                    <p class="description">Generate a dynamic XML sitemap to help search engines index your content more efficiently.</p>
                    
                    <div class="sekhlo-toggle-row">
                        <label>
                            <input type="checkbox" name="sekhlo_sitemap_enabled" value="1" <?php checked($sitemap_enabled, 1); ?>>
                            <strong>Enable XML Sitemap generation</strong>
                        </label>
                    </div>
                    
                    <div class="sekhlo-sitemap-options">
                        <!-- Posts Row -->
                        <div class="sekhlo-sitemap-row">
                            <div class="sekhlo-sitemap-row-left">
                                <label>
                                    <input type="checkbox" name="sekhlo_sitemap_include_posts" value="1" <?php checked($sitemap_posts, 1); ?>>
                                    Posts
                                </label>
                            </div>
                            <div class="sekhlo-option-actions">
                                <a href="<?php echo admin_url('edit.php'); ?>" target="_blank">Edit Posts ↗</a>
                                <a href="<?php echo get_post_type_archive_link('post') ?: home_url('/'); ?>" target="_blank">View Archive ↗</a>
                            </div>
                        </div>

                        <!-- Pages Row -->
                        <div class="sekhlo-sitemap-row">
                            <div class="sekhlo-sitemap-row-left">
                                <label>
                                    <input type="checkbox" name="sekhlo_sitemap_include_pages" value="1" <?php checked($sitemap_pages, 1); ?>>
                                    Pages
                                </label>
                            </div>
                            <div class="sekhlo-option-actions">
                                <a href="<?php echo admin_url('edit.php?post_type=page'); ?>" target="_blank">Edit Pages ↗</a>
                                <a href="<?php echo home_url('/'); ?>" target="_blank">View Site ↗</a>
                            </div>
                        </div>

                        <!-- Categories Row -->
                        <div class="sekhlo-sitemap-row">
                            <div class="sekhlo-sitemap-row-left">
                                <label>
                                    <input type="checkbox" name="sekhlo_sitemap_include_categories" value="1" <?php checked($sitemap_cats, 1); ?>>
                                    Categories
                                </label>
                            </div>
                            <div class="sekhlo-option-actions">
                                <a href="<?php echo admin_url('edit-tags.php?taxonomy=category'); ?>" target="_blank">Edit Categories ↗</a>
                                <a href="<?php echo admin_url('edit-tags.php?taxonomy=category'); ?>" target="_blank">View List ↗</a>
                            </div>
                        </div>
                    </div>
                    
                    <p class="description">Select which content types to include in your public XML sitemap. Use the links to manage your content.</p>
                </div>
                
                <p class="submit"><button type="submit" name="sekhlo_save_settings" class="button button-primary button-large">Save AI & SEO Settings</button></p>


                <!-- robots.txt Section -->
                <div class="sekhlo-settings-card">
                    <h2>
                        🤖 robots.txt Editor
                        <a href="<?php echo home_url('/robots.txt'); ?>" target="_blank" class="button button-small">View File ↗</a>
                    </h2>
                    <p class="description">Edit your website's robots.txt file to control search engine crawling.</p>
                    
                    <textarea name="sekhlo_robots_content" class="sekhlo-code-editor"><?php echo esc_textarea($robots_content); ?></textarea>
                    
                    <p class="submit"><button type="submit" name="sekhlo_save_robots" class="button button-primary button-large">Save robots.txt</button></p>
                </div>

                <!-- .htaccess Section -->
                <div class="sekhlo-settings-card">
                    <h2>⚙️ .htaccess Editor</h2>
                    <div class="sekhlo-warning">
                        <strong>⚠️ Critical Warning:</strong> Editing .htaccess incorrectly can break your website. A backup will be created automatically before saving.
                    </div>
                    
                    <textarea name="sekhlo_htaccess_content" class="sekhlo-code-editor"><?php echo esc_textarea($htaccess_content); ?></textarea>
                    
                    <p class="submit"><button type="submit" name="sekhlo_save_htaccess" class="button button-primary button-large" onclick="return confirm('Are you sure? This can break your website if edited incorrectly.');">Save .htaccess</button></p>
                </div>

            </form>
        </div>
        <?php
    }
}

// Initialize
new Sekhlo_Settings_Page();
