<?php
/**
 * Core Schema Manager Class
 *
 * Handles schema CRUD operations, post type detection, and data migration.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Sekhlo_Schema_Manager {

    /**
     * Meta key for storing schemas.
     */
    const META_KEY = 'sekhlo_schema_code_data';
    const SEO_TITLE_KEY = 'sekhlo_seo_title';
    const SEO_DESC_KEY = 'sekhlo_seo_description';
    const SEO_IMG_KEY = 'sekhlo_seo_image';
    const SEO_KEYPHRASES_KEY = 'sekhlo_seo_keyphrases';
    const SEO_ROBOTS_INDEX_KEY = 'sekhlo_seo_robots_index';
    const SEO_ROBOTS_FOLLOW_KEY = 'sekhlo_seo_robots_follow';
    const SEO_ROBOTS_ADVANCED_KEY = 'sekhlo_seo_robots_advanced';
    const SEO_BREADCRUMB_TITLE_KEY = 'sekhlo_seo_breadcrumb_title';
    const SEO_CANONICAL_URL_KEY = 'sekhlo_seo_canonical_url';

    /**
     * Old meta key for migration.
     */
    const OLD_META_KEY = 'sekhlo_schema_code';

    /**
     * Constructor.
     */
    public function __construct() {
        add_action('save_post', array($this, 'save_schemas'));
        add_action('init', array($this, 'register_meta_fields'));
    }

    /**
     * Register meta fields for REST API (Gutenberg compatibility)
     */
    public function register_meta_fields() {
        $post_types = $this->get_supported_post_types();
        
        foreach ($post_types as $post_type) {
            // Register Schemas Data (CRITICAL for Gutenberg)
            // Using 'string' type with JSON for reliability - complex arrays can be problematic in REST API
            register_post_meta($post_type, self::META_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'default' => '',
                // content is already sanitized by the editor/JS before sending
                // using a simple pass-through to ensure we don't accidentally wipe data on save
                'sanitize_callback' => function($value) {
                    return $value;
                },
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            // Register SEO Title
            register_post_meta($post_type, self::SEO_TITLE_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            // Register SEO Description
            register_post_meta($post_type, self::SEO_DESC_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'sanitize_textarea_field',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            // Register other SEO fields
            register_post_meta($post_type, self::SEO_IMG_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'esc_url_raw',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_KEYPHRASES_KEY, array(
                'type' => 'array',
                'single' => true,
                'show_in_rest' => array(
                    'schema' => array(
                        'type' => 'array',
                        'items' => array('type' => 'string')
                    )
                ),
                'sanitize_callback' => function($value) {
                    return array_map('sanitize_text_field', (array)$value);
                },
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_ROBOTS_INDEX_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_ROBOTS_FOLLOW_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_ROBOTS_ADVANCED_KEY, array(
                'type' => 'array',
                'single' => true,
                'show_in_rest' => array(
                    'schema' => array(
                        'type' => 'array',
                        'items' => array('type' => 'string')
                    )
                ),
                'sanitize_callback' => function($value) {
                    return array_map('sanitize_text_field', (array)$value);
                },
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_BREADCRUMB_TITLE_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'sanitize_text_field',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
            
            register_post_meta($post_type, self::SEO_CANONICAL_URL_KEY, array(
                'type' => 'string',
                'single' => true,
                'show_in_rest' => true,
                'sanitize_callback' => 'esc_url_raw',
                'auth_callback' => function() {
                    return current_user_can('edit_posts');
                }
            ));
        }
    }

    /**
     * Sanitize schemas array for saving
     * 
     * @param mixed $value The schemas array or JSON string
     * @return array Sanitized array of schemas
     */
    public function sanitize_schemas($value) {
        // Handle JSON string input (from hidden field)
        if (is_string($value)) {
            $value = json_decode(stripslashes($value), true);
        }
        
        // Ensure it's an array
        if (!is_array($value)) {
            return array();
        }
        
        // Sanitize each schema
        return array_map(function($schema) {
            if (!is_array($schema)) {
                return array();
            }
            return array(
                'id' => isset($schema['id']) ? sanitize_text_field($schema['id']) : '',
                'type' => isset($schema['type']) ? sanitize_text_field($schema['type']) : '',
                'name' => isset($schema['name']) ? sanitize_text_field($schema['name']) : '',
                'code' => isset($schema['code']) ? $schema['code'] : '', // Keep JSON-LD code as-is
                'enabled' => isset($schema['enabled']) ? (bool)$schema['enabled'] : true
            );
        }, $value);
    }
    
    /**
     * Sanitize schemas and return as JSON string (for REST API)
     * 
     * @param mixed $value The schemas array or JSON string
     * @return string JSON encoded sanitized schemas
     */
    public function sanitize_schemas_json($value) {
        // If it's already a valid JSON string with array data, decode it first
        if (is_string($value) && !empty($value)) {
            // Try with stripslashes (standard WP behavior)
            $decoded = json_decode(stripslashes($value), true);
            
            // If that failed, try raw value (sometimes REST API doesn't slash)
            if (json_last_error() !== JSON_ERROR_NONE) {
                $decoded = json_decode($value, true);
            }
            
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $value = $decoded;
            }
        }
        
        // If empty, return empty JSON array
        if (empty($value)) {
            return '[]';
        }
        
        // Sanitize as array
        $sanitized = $this->sanitize_schemas($value);
        
        // Return as JSON string
        return wp_json_encode($sanitized);
    }

    /**
     * Get all public post types.
     *
     * @return array List of supported post types.
     */
    public function get_supported_post_types() {
        $post_types = get_post_types(array('public' => true), 'names');
        // Filter out attachments if needed, but keeping generally open is good.
        return array_values($post_types);
    }

    /**
     * Get schemas for a specific post.
     * Handles migration from old single-string format and JSON string format.
     *
     * @param int $post_id Post ID.
     * @return array List of schema arrays.
     */
    /**
     * Get schemas for a specific post.
     * Handles migration from old single-string format and JSON string format.
     * Robustly handles double-encoded or slashed JSON.
     *
     * @param int $post_id Post ID.
     * @return array List of schema arrays.
     */
    public function get_schemas($post_id) {
        $schemas = get_post_meta($post_id, self::META_KEY, true);
        
        // Handle JSON string format (from REST API / Gutenberg)
        if (is_string($schemas) && !empty($schemas)) {
            // First try basic decode
            $decoded = json_decode($schemas, true);
            
            // If failed, try stripping slashes first
            if (json_last_error() !== JSON_ERROR_NONE) {
                $decoded = json_decode(stripslashes($schemas), true);
            }

            // Handle double-encoded JSON (string inside string)
            if (is_string($decoded)) {
                $decoded_again = json_decode($decoded, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded_again)) {
                    $decoded = $decoded_again;
                }
            }
            
            if (is_array($decoded)) {
                $schemas = $decoded;
            } else {
                // If decoding completely failed, check if it's the old legacy format
                // or just failed. If it failed, don't overwrite variable yet so migration check can run.
                if (json_last_error() === JSON_ERROR_NONE) {
                    $schemas = array();
                }
            }
        }

        // Migration check from old format
        if (empty($schemas)) {
            $old_schema = get_post_meta($post_id, self::OLD_META_KEY, true);
            if (!empty($old_schema)) {
                $schemas = array(
                    array(
                        'id' => uniqid('schema_'),
                        'type' => 'custom',
                        'name' => 'Legacy Schema',
                        'code' => $old_schema,
                        'enabled' => true
                    )
                );
                // Update new meta with JSON format
                update_post_meta($post_id, self::META_KEY, wp_json_encode($schemas));
            }
        }

        return is_array($schemas) ? $schemas : array();
    }

    /**
     * Save schemas for a post (Classic Editor metabox support).
     * Note: Gutenberg saves meta fields directly via REST API using register_post_meta.
     *
     * @param int $post_id Post ID.
     */
    public function save_schemas($post_id) {
        // Basic security check: users must have permission
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Check if our nonce is set (Classic/Metabox Save)
        if (!isset($_POST['sekhlo_schema_nonce'])) {
            return;
        }

        // Verify nonce
        if (!wp_verify_nonce($_POST['sekhlo_schema_nonce'], 'sekhlo_save_schemas')) {
            return;
        }
        
        // Debug
        error_log('Sekhlo: Saving schemas for post ' . $post_id);
        
        // Handle Schema JSON from Classic Editor metabox
        if (isset($_POST['sekhlo_schemas_json'])) {
            $json_schemas = stripslashes($_POST['sekhlo_schemas_json']);
            
            // Sometimes data might be double escaped depending on server/WP config
            if (strpos($json_schemas, '\"') !== false) {
                $json_schemas = stripslashes($json_schemas);
            }

            $schemas = json_decode($json_schemas, true);

            if (is_array($schemas)) {
                $sanitized = $this->sanitize_schemas($schemas);
                // Save as JSON string for consistency with REST API
                update_post_meta($post_id, self::META_KEY, wp_json_encode($sanitized));
            }
        }

        // Save SEO Meta from Classic Editor
        if (isset($_POST['sekhlo_seo_title'])) {
            $title = sanitize_text_field($_POST['sekhlo_seo_title']);
            update_post_meta($post_id, self::SEO_TITLE_KEY, $title);
        }
        
        if (isset($_POST['sekhlo_seo_description'])) {
            $desc = sanitize_textarea_field($_POST['sekhlo_seo_description']);
            update_post_meta($post_id, self::SEO_DESC_KEY, $desc);
        }
        
        // Handle Post Slug Update
        if (isset($_POST['sekhlo_post_slug'])) {
            $new_slug = sanitize_title($_POST['sekhlo_post_slug']);
            if (!empty($new_slug) && $new_slug !== get_post_field('post_name', $post_id)) {
                // Check if slug is unique
                $slug_check = get_page_by_path($new_slug, OBJECT, get_post_type($post_id));
                if (!$slug_check || $slug_check->ID === $post_id) {
                    wp_update_post(array(
                        'ID' => $post_id,
                        'post_name' => $new_slug
                    ));
                }
            }
        }
        
        // Save additional SEO fields
        if (isset($_POST['sekhlo_seo_keyphrases'])) {
            $keyphrases = array_map('sanitize_text_field', explode(',', $_POST['sekhlo_seo_keyphrases']));
            update_post_meta($post_id, self::SEO_KEYPHRASES_KEY, array_slice($keyphrases, 0, 5));
        }
        if (isset($_POST['sekhlo_robots_index'])) {
            update_post_meta($post_id, self::SEO_ROBOTS_INDEX_KEY, sanitize_text_field($_POST['sekhlo_robots_index']));
        }
        if (isset($_POST['sekhlo_robots_follow'])) {
            update_post_meta($post_id, self::SEO_ROBOTS_FOLLOW_KEY, sanitize_text_field($_POST['sekhlo_robots_follow']));
        }
        if (isset($_POST['sekhlo_robots_advanced'])) {
            update_post_meta($post_id, self::SEO_ROBOTS_ADVANCED_KEY, array_map('sanitize_text_field', (array)$_POST['sekhlo_robots_advanced']));
        }
        if (isset($_POST['sekhlo_breadcrumb_title'])) {
            update_post_meta($post_id, self::SEO_BREADCRUMB_TITLE_KEY, sanitize_text_field($_POST['sekhlo_breadcrumb_title']));
        }
        if (isset($_POST['sekhlo_canonical_url'])) {
            update_post_meta($post_id, self::SEO_CANONICAL_URL_KEY, esc_url_raw($_POST['sekhlo_canonical_url']));
        }
    }
}
