jQuery(document).ready(function ($) {

    // Helper function to parse schemas from various sources
    function parseSchemas(data) {
        if (!data) return [];

        // If already an array, return it
        if (Array.isArray(data)) return data;

        // If string, try to parse
        if (typeof data === 'string' && data.length > 0) {
            try {
                // Handle double-encoded JSON
                var parsed = JSON.parse(data);
                if (typeof parsed === 'string') {
                    parsed = JSON.parse(parsed);
                }
                if (Array.isArray(parsed)) return parsed;
            } catch (e) {
                console.log('Sekhlo: Failed to parse schemas', e);
            }
        }
        return [];
    }

    // Initialize schemas - try multiple sources for reliability
    var schemas = [];

    // Source 1: Hidden field (most reliable - PHP renders this directly)
    var hiddenField = $('#sekhlo_schemas_json');
    if (hiddenField.length && hiddenField.val()) {
        schemas = parseSchemas(hiddenField.val());
        console.log('Sekhlo: Loaded schemas from hidden field:', schemas.length);
    }

    // Source 2: sekhloData.schemas (passed via wp_localize_script - very reliable)
    if (schemas.length === 0 && window.sekhloData && sekhloData.schemas) {
        schemas = parseSchemas(sekhloData.schemas);
        console.log('Sekhlo: Loaded schemas from sekhloData:', schemas.length);
    }

    // Source 3: window.schemas (passed via wp_add_inline_script)
    if (schemas.length === 0 && window.schemas) {
        schemas = parseSchemas(window.schemas);
        console.log('Sekhlo: Loaded schemas from window.schemas:', schemas.length);
    }

    // Source 4: Gutenberg data store (for block editor)
    if (schemas.length === 0 && window.wp && wp.data && wp.data.select('core/editor')) {
        try {
            var currentMeta = wp.data.select('core/editor').getEditedPostAttribute('meta');
            if (currentMeta && currentMeta['sekhlo_schema_code_data']) {
                schemas = parseSchemas(currentMeta['sekhlo_schema_code_data']);
                console.log('Sekhlo: Loaded schemas from Gutenberg store:', schemas.length);
            }
        } catch (e) {
            // Silent fail - Gutenberg may not be available
        }
    }

    // Ensure schemas is always an array
    if (!Array.isArray(schemas)) {
        schemas = [];
    }

    // --- Tab Switching ---
    $('.sekhlo-tab').on('click', function () {
        var target = $(this).data('tab');

        $('.sekhlo-tab').removeClass('active');
        $(this).addClass('active');

        $('.sekhlo-tab-content').removeClass('active');
        $('#' + target).addClass('active');

        if (target === 'sekhlo_testing') {
            calculateAIScore();
        }
        if (target === 'sekhlo_internal_links') {
            generateInternalLinkSuggestions();
        }
        if (target === 'sekhlo_preview') {
            updatePreviews();
        }
        if (target === 'sekhlo_meta_seo') {
            updateMetaCounters();
        }
        if (target === 'sekhlo_ai_prompt') {
            // Generate prompt when AI tab is opened if textarea is empty
            var currentPrompt = $('#sekhlo-ai-prompt-textarea').val();
            if (!currentPrompt || currentPrompt.trim() === '') {
                var prompt = generateAIPrompt();
                $('#sekhlo-ai-prompt-textarea').val(prompt);
            }
        }
    });

    // --- Collapsible ---
    // --- Collapsible ---
    $('.sekhlo-collapsible-header').on('click', function () {
        $(this).closest('.sekhlo-collapsible').toggleClass('open');
    });

    // --- Checkbox Tags (Modern) ---
    $('.sekhlo-checkbox-modern input[type="checkbox"]').on('change', function () {
        $(this).closest('.sekhlo-checkbox-modern').toggleClass('active', $(this).is(':checked'));
    });

    // --- Dark Mode ---
    $('#sekhlo-dark-toggle').on('change', function () {
        if ($(this).is(':checked')) {
            $('body').addClass('sekhlo-dark-mode');
            localStorage.setItem('sekhlo_dark_mode', '1');
        } else {
            $('body').removeClass('sekhlo-dark-mode');
            localStorage.setItem('sekhlo_dark_mode', '0');
        }
    });
    // Restore dark mode
    if (localStorage.getItem('sekhlo_dark_mode') === '1') {
        $('#sekhlo-dark-toggle').prop('checked', true).trigger('change');
    }

    // --- Schema Management ---

    // Initial Render
    renderSchemas();

    // Toggle Schema Card Body
    $(document).on('click', '.sekhlo-schema-card-header', function (e) {
        if (!$(e.target).is('input') && !$(e.target).is('button') && !$(e.target).closest('button').length && !$(e.target).closest('.sekhlo-toggle').length) {
            $(this).closest('.sekhlo-schema-card').toggleClass('open');
        }
    });

    // Enabled Toggle
    $(document).on('change', '.schema-enable-toggle', function () {
        var index = $(this).data('index');
        schemas[index].enabled = $(this).is(':checked');
        updateSchemaField();
    });



    // Delete Schema
    $(document).on('click', '.delete-schema', function (e) {
        e.preventDefault();
        e.stopPropagation();
        if (confirm('Are you sure you want to delete this schema?')) {
            var index = $(this).data('index');
            schemas.splice(index, 1);
            updateSchemaField();
            renderSchemas();
        }
    });

    // Helper to extract JSON from string (handles <script> tags)
    function cleanJsonLd(raw) {
        if (!raw) return "";
        var clean = raw.trim();
        // Regex to strip <script> tags and any attributes
        clean = clean.replace(/<script\b[^>]*>|<\/script>/gi, '').trim();
        return clean;
    }

    // Add Schema Button
    $('#add-new-schema').on('click', function () {
        var raw = cleanJsonLd($('#sekhlo-quick-paste-json').val());
        var customName = $('#sekhlo-quick-paste-name').val().trim();
        var msgBox = $('#sekhlo-quick-paste-validation');

        if (!raw) {
            // If empty, create a blank schema
            schemas.push({
                id: 'schema_' + Date.now(),
                type: 'custom',
                name: customName || 'Custom Schema',
                code: '{\n  "@context": "https://schema.org",\n  "@type": "Thing"\n}',
                enabled: true
            });
        } else {
            try {
                var data = JSON.parse(raw);
                var itemsToAdd = Array.isArray(data) ? data : [data];

                itemsToAdd.forEach(function (item) {
                    var type = item['@type'] || 'Custom';
                    var typeLabel = typeof type === 'string' ? type : (Array.isArray(type) ? type[0] : 'Custom');
                    var finalName = customName;

                    if (!finalName || itemsToAdd.length > 1) {
                        finalName = (finalName ? finalName + ' (' + typeLabel + ')' : typeLabel + ' Schema');
                    }

                    schemas.push({
                        id: 'schema_' + Date.now() + '_' + Math.random().toString(36).substr(2, 5),
                        type: 'custom',
                        name: finalName,
                        code: JSON.stringify(item, null, 2),
                        enabled: true
                    });
                });

                // Clear fields
                $('#sekhlo-quick-paste-json').val('');
                $('#sekhlo-quick-paste-name').val('');
                msgBox.empty();

            } catch (e) {
                msgBox.html('<span style="color:#dc2626; font-weight:500;">Invalid JSON - please check your code</span>');
                return;
            }
        }

        updateSchemaField();
        renderSchemas(schemas.length - 1);
    });

    // Template link click
    $(document).on('click', '[data-action="go-templates"]', function (e) {
        e.preventDefault();
        $('.sekhlo-tab[data-tab="sekhlo_templates"]').click();
    });

    // Real-time Validation
    $(document).on('input', '#sekhlo-quick-paste-json', function () {
        var code = cleanJsonLd($(this).val());
        var msgBox = $('#sekhlo-quick-paste-validation');

        if (!code) {
            msgBox.empty();
            return;
        }
        try {
            JSON.parse(code);
            msgBox.html('<span style="color:#10b981; font-weight:500;">Valid JSON</span>');
        } catch (e) {
            msgBox.html('<span style="color:#dc2626; font-weight:500;">Invalid JSON</span>');
        }
    });

    // --- Drag and Drop Removed for Simplicity ---

    // --- Bulk Actions ---
    $('#sekhlo-select-all').on('change', function () {
        $('.schema-select').prop('checked', $(this).is(':checked'));
    });

    $('#sekhlo-apply-bulk').on('click', function () {
        var action = $('#sekhlo-bulk-action-select').val();
        if (!action) return;

        if (action === 'delete') {
            if (!confirm('Are you sure you want to delete ALL schemas?')) return;
            schemas = [];
        } else if (action === 'enable') {
            schemas.forEach(function (s) { s.enabled = true; });
        } else if (action === 'disable') {
            schemas.forEach(function (s) { s.enabled = false; });
        }

        updateSchemaField();
        renderSchemas();
    });

    // --- Import / Export ---
    $('#sekhlo-export-all-btn').on('click', function () {
        if (schemas.length === 0) return;
        var dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(schemas, null, 2));
        var downloadAnchorNode = document.createElement('a');
        downloadAnchorNode.setAttribute("href", dataStr);
        downloadAnchorNode.setAttribute("download", "sekhlo-schemas-" + sekhloData.postTitle.replace(/\s+/g, '-').toLowerCase() + ".json");
        document.body.appendChild(downloadAnchorNode);
        downloadAnchorNode.click();
        downloadAnchorNode.remove();
    });

    $('#sekhlo-import-btn').on('click', function () {
        $('#sekhlo-import-file').click();
    });

    $('#sekhlo-import-file').on('change', function (e) {
        var file = e.target.files[0];
        if (!file) return;

        var reader = new FileReader();
        reader.onload = function (e) {
            try {
                var imported = JSON.parse(e.target.result);
                if (Array.isArray(imported)) {
                    // Reset IDs to avoid duplicates if importing into same site
                    imported.forEach(s => s.id = 'schema_' + Date.now() + '_' + Math.random().toString(36).substr(2, 5));
                    schemas = schemas.concat(imported);
                    updateSchemaField();
                    renderSchemas();
                    alert('Successfully imported ' + imported.length + ' schemas.');
                } else {
                    alert('Invalid file format. Expected a JSON array.');
                }
            } catch (err) {
                alert('Error parsing JSON file.');
            }
        };
        reader.readAsText(file);
    });

    // Export individual schema
    $(document).on('click', '.export-single-schema', function (e) {
        e.preventDefault();
        var index = $(this).data('index');
        var single = [schemas[index]];
        var dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(single, null, 2));
        var downloadAnchorNode = document.createElement('a');
        downloadAnchorNode.setAttribute("href", dataStr);
        downloadAnchorNode.setAttribute("download", schemas[index].name.replace(/\s+/g, '-').toLowerCase() + ".json");
        document.body.appendChild(downloadAnchorNode);
        downloadAnchorNode.click();
        downloadAnchorNode.remove();
    });

    // --- Template Insertion ---
    $('.sekhlo-template-card').on('click', function () {
        var templateKey = $(this).data('key');
        var template = sekhloTemplates[templateKey];

        if (template) {
            var code = template.code;

            code = code.replace(/%title%/g, sekhloData.postTitle);
            code = code.replace(/%post_url%/g, sekhloData.postUrl);
            code = code.replace(/%site_title%/g, sekhloData.siteTitle);
            code = code.replace(/%site_url%/g, sekhloData.siteUrl);

            var newSchema = {
                id: 'schema_' + Date.now(),
                type: templateKey,
                name: template.label,
                code: code,
                enabled: true
            };

            schemas.push(newSchema);
            updateSchemaField();
            renderSchemas(schemas.length - 1);
            $('.sekhlo-tab[data-tab="sekhlo_schemas"]').click();
        }
    });

    // --- Editor Changes ---
    $(document).on('input propertychange', '.sekhlo-code-editor', function () {
        var index = $(this).data('index');

        // Only update if it's an existing schema item
        if (typeof index !== 'undefined') {
            schemas[index].code = $(this).val();
            updateSchemaField();
            validateJson(index);
        }
    });

    $(document).on('input', '.schema-name-input', function () {
        var index = $(this).data('index');
        schemas[index].name = $(this).val();
        $(this).closest('.sekhlo-schema-card').find('.sekhlo-schema-name').text($(this).val());
        updateSchemaField();
    });

    // --- AI Score Calculation ---
    function calculateAIScore() {
        var score = 0;
        var totalPossible = 0;
        var tips = [];

        if (schemas.length === 0) {
            $('#sekhlo-score-feedback').text('Add schemas to evaluate your score.');
            updateScoreUI(0);
            return;
        }

        // Checklist for AI
        var hasArticle = false;
        var hasFAQ = false;
        var hasSpeakable = false;
        var hasAbout = false;
        var hasMentions = false;
        var hasAuthorExtras = false;

        schemas.forEach(s => {
            if (!s.enabled) return;
            var code = s.code.toLowerCase();
            if (code.includes('"article"') || code.includes('"blogposting"')) hasArticle = true;
            if (code.includes('"faqpage"')) hasFAQ = true;
            if (code.includes('"speakablespecification"')) hasSpeakable = true;
            if (code.includes('"about"')) hasAbout = true;
            if (code.includes('"mentions"')) hasMentions = true;
            if (code.includes('"sameas"') || code.includes('"knowsabout"')) hasAuthorExtras = true;
        });

        if (hasArticle) score += 20; else tips.push('Use "Article" schema for knowledge graph entry.');
        if (hasFAQ) score += 20; else tips.push('Add "FAQ" schema to answer niche-specific questions.');
        if (hasSpeakable) score += 20; else tips.push('Enable "Speakable" (AEO) to capture voice search traffic.');
        if (hasAbout && hasMentions) score += 20; else tips.push('Define "About" and "Mentions" entities for Semantic SEO.');
        if (hasAuthorExtras) score += 20; else tips.push('Add "sameAs" links to author profiles to establish E-E-A-T.');

        updateScoreUI(score);

        if (score >= 80) $('#sekhlo-score-feedback').text('Excellent optimization for AI search engines');
        else if (score >= 50) $('#sekhlo-score-feedback').text('Good foundation, room for improvement');
        else $('#sekhlo-score-feedback').text('Consider adding more schema types for better visibility');

        var tipHtml = tips.map(t => `<li style="margin-bottom:5px;">${t}</li>`).join('');
        $('#sekhlo-score-tips').html(tipHtml);
    }

    function updateScoreUI(score) {
        $('#sekhlo-score-num').text(score);
        $('#sekhlo-score-fill').css('width', score + '%');
        if (score < 40) $('#sekhlo-score-fill').css('background', '#dc2626');
        else if (score < 70) $('#sekhlo-score-fill').css('background', '#f59e0b');
        else $('#sekhlo-score-fill').css('background', '#10b981');
    }



    // --- Helper Functions ---

    function validateJson(index) {
        var code = schemas[index].code;
        var msgBox = $('#validation-' + index);
        try {
            JSON.parse(code);
            msgBox.html('<span style="color:#10b981; font-size:12px; font-weight:500;">Valid JSON</span>');
        } catch (e) {
            msgBox.html('<span style="color:#dc2626; font-size:12px; font-weight:500;">Invalid JSON</span>');
        }
    }

    function updateSchemaField() {
        // Update the hidden field for Classic Editor
        var jsonStr = JSON.stringify(schemas);
        $('#sekhlo_schemas_json').val(jsonStr);

        // Gutenberg Logic
        if (window.wp && wp.data && wp.data.select('core/editor')) {
            // 1. Sync to Gutenberg store
            syncMetaFieldsToGutenberg();

            // Note: We keep the hidden field ENABLED even in Gutenberg.
            // This ensures that if the REST API save fails or is ignored,
            // the classic POST submission (which happens on "Update") will catch it.
        }
    }

    // Sync before Save (Classic & Gutenberg)
    function syncBeforeSave() {
        var raw = cleanJsonLd($('#sekhlo-quick-paste-json').val());
        if (raw) {
            try {
                var data = JSON.parse(raw);
                var customName = $('#sekhlo-quick-paste-name').val().trim();
                var itemsToAdd = Array.isArray(data) ? data : [data];

                itemsToAdd.forEach(function (item) {
                    var type = item['@type'] || 'Custom';
                    var name = customName || (typeof type === 'string' ? type : 'Custom') + ' Schema';

                    schemas.push({
                        id: 'schema_' + Date.now() + '_' + Math.random().toString(36).substr(2, 5),
                        type: 'custom',
                        name: name,
                        code: JSON.stringify(item, null, 2),
                        enabled: true
                    });
                });

                // Clear paste fields so they don't get double added if save is interrupted
                $('#sekhlo-quick-paste-json').val('');
                $('#sekhlo-quick-paste-name').val('');
            } catch (e) {
                // If invalid, we just don't add it automatically
            }
        }
        updateSchemaField();
    }

    // Hook into form submission for classic editor
    $('form#post').on('submit', function () {
        syncBeforeSave();
    });

    // Hook into Gutenberg save (check for wp.data availability)
    if (window.wp && wp.data) {
        var wasSaving = false;
        wp.data.subscribe(function () {
            var isSaving = wp.data.select('core/editor').isSavingPost();
            var isAutosaving = wp.data.select('core/editor').isAutosavingPost();

            if (isSaving && !isAutosaving && !wasSaving) {
                wasSaving = true;
                syncBeforeSave();
                syncMetaFieldsToGutenberg();
            }

            if (!isSaving && wasSaving) {
                wasSaving = false;
            }
        });
    }

    // Sync meta fields to Gutenberg's meta store
    function syncMetaFieldsToGutenberg() {
        // Only run if Gutenberg editor is available
        if (!window.wp || !wp.data || !wp.data.dispatch || !wp.data.select) {
            return;
        }

        try {
            var editor = wp.data.select('core/editor');
            if (!editor) {
                return;
            }

            var metaTitle = $('#sekhlo_seo_title').val() || '';
            var metaDesc = $('#sekhlo_seo_description').val() || '';

            // Prepare schemas array - ensure all items are properly formatted
            var cleanSchemas = schemas.map(function (schema) {
                return {
                    id: schema.id || '',
                    type: schema.type || 'custom',
                    name: schema.name || '',
                    code: schema.code || '',
                    enabled: Boolean(schema.enabled)
                };
            });

            // Convert schemas to JSON string since meta is registered as 'string' type
            var schemasJson = JSON.stringify(cleanSchemas);

            // Update meta fields in Gutenberg's store
            wp.data.dispatch('core/editor').editPost({
                meta: {
                    'sekhlo_seo_title': metaTitle,
                    'sekhlo_seo_description': metaDesc,
                    'sekhlo_schema_code_data': schemasJson
                }
            });
        } catch (e) {
            // Silent fail - Gutenberg may not be fully loaded
        }
    }

    // Sync when meta fields change (real-time sync for Gutenberg)
    $('#sekhlo_seo_title, #sekhlo_seo_description').on('input change blur', function () {
        syncMetaFieldsToGutenberg();
    });

    function renderSchemas(openIndex = null) {
        var container = $('#sekhlo-schema-list-container');
        container.empty();

        if (schemas.length === 0) {
            // Show empty state message
            container.html('<div class="sekhlo-empty-state"><span class="dashicons dashicons-editor-code"></span><p>Your schemas will appear here</p></div>');
            return;
        }

        // Add bulk actions bar when schemas exist
        var bulkHtml = `
            <div class="sekhlo-bulk-actions">
                <select id="sekhlo-bulk-action-select">
                    <option value="">Bulk Actions</option>
                    <option value="enable">Enable All</option>
                    <option value="disable">Disable All</option>
                    <option value="delete">Delete All</option>
                </select>
                <button type="button" id="sekhlo-apply-bulk" class="sekhlo-btn">Apply</button>
                <div style="margin-left:auto; display:flex; gap:8px;">
                    <button type="button" id="sekhlo-import-btn" class="sekhlo-btn" title="Import JSON-LD"><span class="dashicons dashicons-upload"></span></button>
                    <button type="button" id="sekhlo-export-all-btn" class="sekhlo-btn" title="Export All Schemas"><span class="dashicons dashicons-download"></span></button>
                </div>
            </div>
        `;
        container.append(bulkHtml);

        schemas.forEach(function (schema, index) {
            var isOpen = (index === openIndex) ? 'open' : '';
            var isEnabled = schema.enabled ? 'checked' : '';
            var statusClass = schema.enabled ? 'status-enabled' : 'status-disabled';
            var html = `
                <div class="sekhlo-schema-card ${isOpen}">
                    <div class="sekhlo-schema-card-header" data-index="${index}">
                        <div class="sekhlo-schema-info">
                            <span class="sekhlo-schema-badge">${schema.type}</span>
                            <span class="sekhlo-schema-name">${schema.name}</span>
                            <span class="sekhlo-status-dot ${statusClass}" title="${schema.enabled ? 'Enabled' : 'Disabled'}"></span>
                        </div>
                        <div class="sekhlo-schema-controls">
                            <label class="sekhlo-toggle" title="Enable/Disable">
                                <input type="checkbox" class="schema-enable-toggle" data-index="${index}" ${isEnabled}>
                                <span class="sekhlo-slider"></span>
                            </label>
                            <button type="button" class="sekhlo-icon-btn delete-schema" data-index="${index}" title="Delete">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                            <span class="sekhlo-expand-icon dashicons dashicons-arrow-down-alt2"></span>
                        </div>
                    </div>
                    <div class="sekhlo-schema-card-body">
                        <div class="sekhlo-form-group">
                            <label>Schema Name</label>
                            <input type="text" class="sekhlo-input schema-name-input" data-index="${index}" value="${schema.name}">
                        </div>
                        <div class="sekhlo-form-group">
                            <label>JSON-LD Code</label>
                            <textarea class="sekhlo-code-editor" data-index="${index}">${schema.code}</textarea>
                            <div class="validation-msg" id="validation-${index}"></div>
                        </div>
                    </div>
                </div>
            `;
            container.append(html);
            validateJson(index);
        });
    }

    function resolveVariables(text) {
        if (!text) return "";
        var res = text;
        res = res.replace(/%%title%%/g, sekhloData.postTitle);
        res = res.replace(/%%sitename%%/g, sekhloData.siteTitle);
        res = res.replace(/%%sep%%/g, "-");
        res = res.replace(/%%currentyear%%/g, new Date().getFullYear());
        res = res.replace(/%%excerpt%%/g, sekhloData.postContent.substring(0, 160));
        return res;
    }

    function updatePreviews() {
        var serpContainer = $('#sekhlo-preview-content');
        var rawOutput = $('#sekhlo-raw-json-output');
        if (!serpContainer.length) return;

        // --- 1. Update SERP Preview ---
        var title = sekhloData.postTitle;
        var desc = sekhloData.postContent.substring(0, 160) + "...";
        var date = new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
        var favicon = "https://www.google.com/s2/favicons?domain=" + sekhloData.siteUrl + "&sz=32";
        var imgUrl = sekhloData.featuredImageUrl;

        var serpTemplate = `
            <div class="google-serp">
                <div style="display:flex; align-items:center; gap:10px; margin-bottom:10px;">
                    <img src="${favicon}" style="width:26px; height:26px; border-radius:50%; background:#f1f3f4; padding:2px;">
                    <div>
                        <div style="font-size:14px; color:#202124; line-height:1.2;">${sekhloData.siteTitle}</div>
                        <div style="font-size:12px; color:#4d5156; line-height:1.2;">${sekhloData.siteUrl.replace(/^https?:\/\/|^\/\//, '').replace(/\/$/, '')} › ...</div>
                    </div>
                </div>
                <a href="#" class="serp-title">${title}</a>
                <div class="serp-content-flex" style="display:flex; gap:15px; margin-top:5px;">
                    <div style="flex:1;">
                        <div class="serp-desc">
                            <span style="color:#10b981; font-weight:600;">[SEO OK]</span> 
                            <span style="color:#70757a">${date} — </span>
                            ${desc}
                        </div>
                    </div>
                    ${imgUrl ? `<div class="serp-image" style="width:104px; height:104px;"><img src="${imgUrl}" style="width:100%; height:100%; object-fit:cover; border-radius:8px;"></div>` : ''}
                </div>
            </div>
        `;
        serpContainer.html(serpTemplate);

        // --- 2. Update Raw JSON-LD ---
        var combinedCode = "";
        schemas.forEach(function (s) {
            if (s.enabled) {
                var cleanCode = cleanJsonLd(s.code);
                combinedCode += `<script type="application/ld+json">\n${cleanCode}\n<\/script>\n\n`;
            }
        });

        if (rawOutput.length) {
            rawOutput.val(combinedCode || "No enabled schemas to show. Enable some schemas in the 'Schemas' tab first.");
        }
    }

    // --- Copy to Clipboard ---
    $(document).on('click', '#sekhlo-copy-raw-btn', function () {
        var copyText = document.getElementById("sekhlo-raw-json-output");
        if (!copyText || !copyText.value || copyText.value.startsWith("No enabled")) return;

        copyText.select();
        copyText.setSelectionRange(0, 99999); // For mobile

        try {
            var successful = document.execCommand('copy');
            var msg = $('#sekhlo-copy-msg');
            if (successful) {
                msg.text('Copied successfully').css('color', '#10b981').show();
                setTimeout(() => msg.fadeOut(), 2000);
            }
        } catch (err) {
            console.error('Copy failed', err);
        }
    });

    // --- Internal Link Suggestions ---
    function generateInternalLinkSuggestions() {
        var list = $('#sekhlo-internal-suggestions-list');

        // Simulated scan of content for topics
        var content = sekhloData.postContent.toLowerCase();
        var commonTopics = ['marketing', 'seo', 'automation', 'email', 'growth', 'strategy', 'optimization'];
        var foundTopics = commonTopics.filter(topic => content.includes(topic));

        // Simulated suggestions based on found topics
        setTimeout(function () {
            list.empty();
            if (foundTopics.length === 0) {
                list.html('<div style="text-align:center; padding:20px;">No specific topics detected. Try adding more content to get suggestions.</div>');
                return;
            }

            var suggestions = [
                { title: "Top 10 SEO Strategies for 2026", cat: "SEO", relevance: "95%" },
                { title: "Email Marketing Automation Masterclass", cat: "Marketing", relevance: "88%" },
                { title: "How to Build a Growth Engine", cat: "Growth", relevance: "82%" },
                { title: "The Ultimate Guide to CRM Integration", cat: "Automation", relevance: "75%" }
            ];

            // Filter suggestions that match found topics
            var filtered = suggestions.filter(s => foundTopics.some(t => s.title.toLowerCase().includes(t.toLowerCase()) || s.cat.toLowerCase().includes(t.toLowerCase())));

            if (filtered.length === 0) filtered = [suggestions[0], suggestions[1]]; // Fallback

            filtered.forEach(function (s) {
                list.append(`
                    <div class="sekhlo-suggestion-card">
                        <div class="suggestion-info">
                            <a href="#" class="suggestion-title">${s.title}</a>
                            <div class="suggestion-meta">Category: ${s.cat} | Relevance: ${s.relevance}</div>
                        </div>
                        <div class="suggestion-action">
                            <button type="button" class="sekhlo-btn">Copy Link</button>
                        </div>
                    </div>
                `);
            });
        }, 1200);
    }

    // --- Meta Title & Description Functions ---
    function updateMetaCounters() {
        var title = $('#sekhlo_seo_title').val();
        var desc = $('#sekhlo_seo_description').val();

        // Update title counter
        var titleLen = title.length;
        $('#sekhlo-title-counter').text(titleLen + ' characters');

        if (titleLen === 0) {
            $('#sekhlo-title-status').text('').css('color', '#6b7280');
        } else if (titleLen < 50) {
            $('#sekhlo-title-status').text('Too short').css('color', '#f59e0b');
        } else if (titleLen <= 60) {
            $('#sekhlo-title-status').text('Perfect length').css('color', '#10b981');
        } else if (titleLen <= 70) {
            $('#sekhlo-title-status').text('A bit long').css('color', '#f59e0b');
        } else {
            $('#sekhlo-title-status').text('Too long').css('color', '#dc2626');
        }

        // Update description counter
        var descLen = desc.length;
        $('#sekhlo-desc-counter').text(descLen + ' characters');

        if (descLen === 0) {
            $('#sekhlo-desc-status').text('').css('color', '#6b7280');
        } else if (descLen < 120) {
            $('#sekhlo-desc-status').text('Too short').css('color', '#f59e0b');
        } else if (descLen <= 160) {
            $('#sekhlo-desc-status').text('Perfect length').css('color', '#10b981');
        } else if (descLen <= 200) {
            $('#sekhlo-desc-status').text('A bit long').css('color', '#f59e0b');
        } else {
            $('#sekhlo-desc-status').text('Too long').css('color', '#dc2626');
        }

        // Update preview
        updateMetaPreview();
    }

    function updateMetaPreview() {
        var title = $('#sekhlo_seo_title').val() || sekhloData.postTitle;
        var desc = $('#sekhlo_seo_description').val() || 'Add a meta description to see how it appears in search results...';
        var slug = $('#sekhlo_post_slug').val() || sekhloData.postTitle.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');

        // Update preview elements
        $('#sekhlo-preview-title').text(title);
        $('#sekhlo-preview-desc').text(desc);

        // Update URL preview
        var baseUrl = sekhloData.siteUrl.replace(/\/$/, '');
        $('#sekhlo-preview-url').text(baseUrl + '/' + slug + '/');
    }

    function sanitizeSlug(slug) {
        return slug
            .toLowerCase()
            .replace(/[^a-z0-9\s-]/g, '') // Remove special characters
            .replace(/\s+/g, '-')          // Replace spaces with hyphens
            .replace(/-+/g, '-')           // Replace multiple hyphens with single
            .replace(/^-+|-+$/g, '');      // Remove leading/trailing hyphens
    }

    // Real-time updates for meta fields
    $('#sekhlo_seo_title, #sekhlo_seo_description').on('input', function () {
        updateMetaCounters();
    });

    // Real-time slug sanitization and preview update
    $('#sekhlo_post_slug').on('input', function () {
        var sanitized = sanitizeSlug($(this).val());
        if ($(this).val() !== sanitized) {
            var cursorPos = this.selectionStart;
            $(this).val(sanitized);
            this.setSelectionRange(cursorPos, cursorPos);
        }
        updateMetaPreview();
    });

    // Initialize counters on page load
    updateMetaCounters();

    // --- AI Prompt Generator ---
    $(document).on('click', '#sekhlo-generate-ai-prompt', function (e) {
        e.preventDefault();
        console.log('=== Generate AI Prompt button clicked ===');
        console.log('sekhloData available:', typeof sekhloData !== 'undefined');
        if (typeof sekhloData !== 'undefined') {
            console.log('sekhloData:', sekhloData);
        }

        // Generate the prompt first
        var prompt = generateAIPrompt();
        console.log('Generated prompt length:', prompt.length);
        console.log('First 200 chars:', prompt.substring(0, 200));

        // Switch to AI Prompt tab first
        console.log('Switching to AI Prompt tab...');
        $('.sekhlo-tab[data-tab="sekhlo_ai_prompt"]').trigger('click');

        // Set the prompt in textarea after a small delay to ensure tab is switched
        setTimeout(function () {
            console.log('Setting prompt in textarea...');
            var textarea = $('#sekhlo-ai-prompt-textarea');
            console.log('Textarea found:', textarea.length > 0);
            textarea.val(prompt);
            console.log('Textarea value after setting:', textarea.val().length);
        }, 100);
    });

    // --- Meta Tags AI Prompt Generator ---
    $(document).on('click', '#sekhlo-generate-meta-prompt', function (e) {
        e.preventDefault();
        console.log('=== Generate Meta Prompt button clicked ===');

        // Generate the meta prompt
        var prompt = generateMetaTagsPrompt();
        console.log('Generated meta prompt length:', prompt.length);

        // Show the prompt container and set the value
        $('#sekhlo-meta-prompt-container').show();
        $('#sekhlo-meta-prompt-textarea').val(prompt);

        // Scroll to the prompt
        $('html, body').animate({
            scrollTop: $('#sekhlo-meta-prompt-container').offset().top - 100
        }, 500);
    });

    $(document).on('click', '#sekhlo-copy-meta-prompt-btn', function () {
        var promptText = document.getElementById('sekhlo-meta-prompt-textarea');
        if (promptText && promptText.value) {
            promptText.select();
            promptText.setSelectionRange(0, 99999);

            try {
                document.execCommand('copy');
                $('#sekhlo-meta-copy-msg').text('✓ Copied to clipboard!').fadeIn().delay(2000).fadeOut();
            } catch (err) {
                console.error('Copy failed:', err);
                $('#sekhlo-meta-copy-msg').text('Please manually copy the text').fadeIn().delay(3000).fadeOut();
            }
        } else {
            console.error('Textarea not found or empty');
            $('#sekhlo-meta-copy-msg').text('No prompt to copy').css('color', '#dc2626').fadeIn().delay(2000).fadeOut();
        }
    });

    // --- Internal Links AI Prompt Generator ---
    $(document).on('click', '#sekhlo-generate-internal-links-prompt', function (e) {
        e.preventDefault();
        console.log('=== Generate Internal Links Prompt button clicked ===');

        // Generate the internal links prompt
        var prompt = generateInternalLinksPrompt();
        console.log('Generated internal links prompt length:', prompt.length);

        // Show the prompt container and set the value
        $('#sekhlo-internal-links-prompt-container').show();
        $('#sekhlo-internal-links-prompt-textarea').val(prompt);

        // Scroll to the prompt
        $('html, body').animate({
            scrollTop: $('#sekhlo-internal-links-prompt-container').offset().top - 100
        }, 500);
    });

    $(document).on('click', '#sekhlo-copy-internal-links-prompt-btn', function () {
        var promptText = document.getElementById('sekhlo-internal-links-prompt-textarea');
        if (promptText && promptText.value) {
            promptText.select();
            promptText.setSelectionRange(0, 99999);

            try {
                document.execCommand('copy');
                $('#sekhlo-internal-links-copy-msg').text('✓ Copied to clipboard!').fadeIn().delay(2000).fadeOut();
            } catch (err) {
                console.error('Copy failed:', err);
                $('#sekhlo-internal-links-copy-msg').text('Please manually copy the text').fadeIn().delay(3000).fadeOut();
            }
        } else {
            console.error('Textarea not found or empty');
            $('#sekhlo-internal-links-copy-msg').text('No prompt to copy').css('color', '#dc2626').fadeIn().delay(2000).fadeOut();
        }
    });

    $(document).on('click', '#sekhlo-copy-ai-prompt-btn', function () {
        var promptText = document.getElementById('sekhlo-ai-prompt-textarea');
        if (promptText && promptText.value) {
            promptText.select();
            promptText.setSelectionRange(0, 99999);

            try {
                document.execCommand('copy');
                $('#sekhlo-ai-copy-msg').text('✓ Copied to clipboard!').fadeIn().delay(2000).fadeOut();
            } catch (err) {
                console.error('Copy failed:', err);
                $('#sekhlo-ai-copy-msg').text('Please manually copy the text').fadeIn().delay(3000).fadeOut();
            }
        } else {
            console.error('Textarea not found or empty');
            $('#sekhlo-ai-copy-msg').text('No prompt to copy').css('color', '#dc2626').fadeIn().delay(2000).fadeOut();
        }
    });

    function generateAIPrompt() {
        console.log('=== generateAIPrompt function called ===');

        // Check if sekhloData exists
        if (typeof sekhloData === 'undefined') {
            console.error('ERROR: sekhloData is undefined');
            return 'Error: Page data not available. Please refresh the page and try again.';
        }

        console.log('sekhloData found:', sekhloData);

        var postTitle = sekhloData.postTitle || 'Your Page Title';
        var postUrl = sekhloData.postUrl || window.location.href;
        var postContent = sekhloData.postContent ? sekhloData.postContent.substring(0, 1000) : 'No content available';
        var siteTitle = sekhloData.siteTitle || 'Your Website';
        var siteUrl = sekhloData.siteUrl || window.location.origin;
        var featuredImage = sekhloData.featuredImageUrl || '';

        // Get current meta title and description from input fields (in case user has entered them)
        var metaTitle = $('#sekhlo_seo_title').val() || sekhloData.metaTitle || '';
        var metaDescription = $('#sekhlo_seo_description').val() || sekhloData.metaDescription || '';

        console.log('Prompt data prepared:', {
            postTitle: postTitle,
            postUrl: postUrl,
            siteTitle: siteTitle,
            siteUrl: siteUrl,
            contentLength: postContent.length,
            hasFeaturedImage: !!featuredImage,
            hasMetaTitle: !!metaTitle,
            hasMetaDescription: !!metaDescription
        });

        var prompt = `# Schema.org JSON-LD Generator (SEO + Structured Data Optimization)
You are an SEO specialist and structured data expert. Your task is to generate comprehensive, valid, and SEO-optimized Schema.org JSON-LD markup for a webpage.

🔹 Page Information
- **Page Title:** ${postTitle}
- **Page URL:** ${postUrl}
- **Website Name:** ${siteTitle}
- **Website URL:** ${siteUrl}
${featuredImage ? '- **Featured Image:** ' + featuredImage : ''}
${metaTitle ? '- **Meta Title:** ' + metaTitle : ''}
${metaDescription ? '- **Meta Description:** ' + metaDescription : ''}

🔹 Content Summary:
${postContent}

🔹 Your Task
Analyze the page content and generate the most appropriate Schema.org JSON-LD structured data that will:
✓ Improve search engine understanding of the content
✓ Enable rich snippets and enhanced search results
✓ Support voice search and AI-powered search engines
✓ Follow Google's structured data guidelines
✓ Pass validation in Google's Rich Results Test

🔹 Schema Types to Consider
Based on the content, include the most relevant schema types:
- **Article/BlogPosting** - For blog posts and articles
- **WebPage** - For general web pages
- **Organization** - For business/company information
- **Person** - For author and people profiles
- **BreadcrumbList** - For navigation hierarchy
- **FAQPage** - For Q&A content
- **HowTo** - For tutorials and guides
- **Product/Service** - For products or services
- **Review/AggregateRating** - For reviews and ratings
- **VideoObject** - For video content
- **Event** - For events and webinars
- **LocalBusiness** - For local businesses
- Any other relevant schema types

🔹 Required Properties (Include When Applicable)
**For Article/BlogPosting:**
- headline, description, image, author, publisher, datePublished, dateModified, mainEntityOfPage

**For Organization:**
- name, url, logo, sameAs, contactPoint, address

**For Person:**
- name, url, image, jobTitle, worksFor, sameAs

**For WebPage:**
- name, description, url, breadcrumb, mainEntity

🔹 SEO & AI Optimization Features
✓ Include "speakable" property for voice search optimization
✓ Add "about" and "mentions" for entity recognition and semantic SEO
✓ Include "keywords" for topic relevance
✓ Add "inLanguage" for language specification
✓ Use "sameAs" for entity disambiguation and social profiles
✓ Include "potentialAction" for search actions when relevant
✓ Add "mainEntityOfPage" to establish primary content
✓ Use proper image schema with width, height, and caption

🔹 Technical Requirements
✓ Use valid JSON-LD syntax (proper quotes, commas, brackets)
✓ Include "@context": "https://schema.org"
✓ Use proper data types (Text, URL, Date in ISO 8601 format)
✓ Ensure all URLs are absolute (include full domain)
✓ Validate against Schema.org specifications
✓ Make it compatible with Google's Rich Results Test
✓ Use proper nesting for complex objects

🔹 CRITICAL OUTPUT FORMAT RULES

**IMPORTANT:** 
1. Output ONLY pure JSON code
2. DO NOT wrap the JSON in <script> tags
3. DO NOT include <script type="application/ld+json"> or </script>
4. Start directly with the opening curly brace {
5. End directly with the closing curly brace }
6. Each schema should be a separate, complete JSON object
7. If multiple schemas are needed, provide them as separate JSON objects (one after another)

**Example of CORRECT output format:**
\`\`\`json
{
  "@context": "https://schema.org",
  "@type": "Article",
  "headline": "Your Article Title",
  "description": "Article description",
  "image": {
    "@type": "ImageObject",
    "url": "https://example.com/image.jpg",
    "width": 1200,
    "height": 630
  },
  "author": {
    "@type": "Person",
    "name": "Author Name",
    "url": "https://example.com/author"
  },
  "publisher": {
    "@type": "Organization",
    "name": "Publisher Name",
    "logo": {
      "@type": "ImageObject",
      "url": "https://example.com/logo.png"
    }
  },
  "datePublished": "2024-01-15T08:00:00+00:00",
  "dateModified": "2024-01-15T08:00:00+00:00",
  "mainEntityOfPage": {
    "@type": "WebPage",
    "@id": "https://example.com/article"
  }
}
\`\`\`

**If multiple schemas are needed, provide them separately:**
\`\`\`json
{
  "@context": "https://schema.org",
  "@type": "Article",
  ...
}
\`\`\`

\`\`\`json
{
  "@context": "https://schema.org",
  "@type": "BreadcrumbList",
  ...
}
\`\`\`

🔹 Quality Checklist
Before providing the output, ensure:
✓ Valid JSON syntax (no trailing commas, proper quotes)
✓ All required properties are included
✓ URLs are absolute and properly formatted
✓ Dates are in ISO 8601 format (YYYY-MM-DDTHH:MM:SS+00:00)
✓ Image objects include width and height when possible
✓ No <script> tags in the output
✓ Schema types match the content accurately
✓ Passes Google's Rich Results Test validation

🎯 Goal
Generate production-ready Schema.org JSON-LD markup that:
✓ Is immediately copy-paste ready for WordPress
✓ Improves search visibility and rich snippets
✓ Follows all Schema.org and Google guidelines
✓ Optimizes for modern search engines and AI

Please analyze the content and generate the appropriate Schema.org JSON-LD markup. Remember: Output ONLY pure JSON without <script> tags.`;

        console.log('Prompt generated successfully. Length:', prompt.length);
        return prompt;
    }

    function generateMetaTagsPrompt() {
        console.log('=== generateMetaTagsPrompt function called ===');

        // Check if sekhloData exists
        if (typeof sekhloData === 'undefined') {
            console.error('ERROR: sekhloData is undefined');
            return 'Error: Page data not available. Please refresh the page and try again.';
        }

        var postTitle = sekhloData.postTitle || 'Your Page Title';
        var postUrl = sekhloData.postUrl || window.location.href;
        var postContent = sekhloData.postContent ? sekhloData.postContent.substring(0, 1500) : 'No content available';
        var siteTitle = sekhloData.siteTitle || 'Your Website';
        var currentSlug = $('#sekhlo_post_slug').val() || sekhloData.postTitle.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');

        var prompt = `# SEO Meta Tags Generator (CTR + Semantic SEO)
You are an SEO strategist and conversion-focused copywriter. Your task is to generate highly optimized, semantic, and click-worthy SEO meta tags for a webpage using the information provided.
Analyze the page for search intent, primary and secondary keywords, semantic relevance, and click-through-rate (CTR) optimization, then generate the best possible meta title, meta description, and URL slug.

🔹 Page Information
- **Current Page Title:** ${postTitle}
- **Page URL:** ${postUrl}
- **Website Name:** ${siteTitle}
- **Current Slug:** ${currentSlug}

🔹 Content Summary:
${postContent}

🔹 Target Audience (Optional but Recommended)
Who is this page for? [Beginners, professionals, buyers, users, developers, etc.]
Problem or need: [What the user is trying to solve]

🔹 Search Intent
Primary intent: [Informational / Navigational / Transactional / Commercial]
Secondary intent (if any): [Optional]

🔹 Optimization Requirements

1️⃣ Meta Title (50–60 characters)
- Must include the primary keyword
- Be clear, compelling, and click-worthy
- Use power words where appropriate (e.g., Best, Complete, Official, Guide, Proven)
- Align with search intent
- Avoid keyword stuffing
- Preferred format (if space allows): Primary Keyword | Secondary Keyword | Brand

2️⃣ Meta Description (150–160 characters)
- Persuasive and benefit-driven
- Include primary + secondary keywords naturally
- Clearly communicate the value proposition
- Include a CTA or action-oriented phrase
- Match search intent and reduce pogo-sticking

3️⃣ URL Slug (SEO-Friendly)
- Clean, readable, and semantic
- 3–5 words ideal
- Use hyphens and lowercase only
- Include the primary keyword
- Avoid unnecessary stop words
- Must align with the title and description

🔹 SEO Strategy Considerations
- Identify primary and secondary keywords
- Optimize for semantic SEO and entities
- Maintain consistency across title, description, and slug
- Prioritize CTR + relevance over exact-match repetition
- Optimize for featured snippets and voice search
- Avoid clickbait or spammy phrasing

🔹 Output Format (Use Exactly)

**Meta Title:**
[Optimized meta title]

**Meta Description:**
[Optimized meta description]

**URL Slug:**
[optimized-slug-here]

**Explanation:**
- Primary keyword(s): [list]
- Secondary keyword(s): [list]
- Search intent: [type]
- CTR and SEO strategy used: [brief explanation]

**Alternative Options:**
- 2–3 alternative meta titles
- 2–3 alternative meta descriptions
- 1–2 alternative URL slugs

🎯 Goal
Generate high-performing SEO meta tags that:
✓ Increase click-through rate
✓ Match user search intent
✓ Strengthen semantic relevance
✓ Perform well in modern Google SERPs

🧠 Optional Enhancers (Use When Needed)
- If competition is high → favor CTR-driven wording
- If brand authority matters → include trust signals
- If freshness matters → include year or timeframe
- If page is YMYL → prioritize clarity and credibility`;

        console.log('Meta prompt generated successfully. Length:', prompt.length);
        return prompt;
    }

    function generateInternalLinksPrompt() {
        console.log('=== generateInternalLinksPrompt function called ===');

        // Check if sekhloData exists
        if (typeof sekhloData === 'undefined') {
            console.error('ERROR: sekhloData is undefined');
            return 'Error: Page data not available. Please refresh the page and try again.';
        }

        var postTitle = sekhloData.postTitle || 'Your Page Title';
        var postUrl = sekhloData.postUrl || window.location.href;
        var postContent = sekhloData.postContent ? sekhloData.postContent.substring(0, 2000) : 'No content available';
        var siteUrl = sekhloData.siteUrl || window.location.origin;
        var sitemapUrl = siteUrl.replace(/\/$/, '') + '/sitemap.xml';

        var prompt = `# Internal Linking Strategy Generator (SEO + User Experience)
You are an SEO specialist and information architecture expert. Your task is to analyze the current page content and suggest the most relevant internal links from the website's sitemap to improve SEO, user experience, and site structure.

🔹 Current Page Information
- **Page Title:** ${postTitle}
- **Page URL:** ${postUrl}
- **Website:** ${siteUrl}
- **Sitemap URL:** ${sitemapUrl}

🔹 Page Content Summary:
${postContent}

🔹 Your Task
1. **Fetch and analyze the sitemap:** Visit ${sitemapUrl} to get a list of all available pages on the website
2. **Analyze the current page content** to understand:
   - Main topic and subtopics
   - Keywords and entities mentioned
   - User intent and journey stage
   - Content gaps that could be filled with internal links
3. **Identify relevant pages** from the sitemap that should be linked from this page
4. **Suggest strategic internal links** that will:
   - Improve SEO through topical relevance
   - Enhance user experience and navigation
   - Distribute page authority (PageRank)
   - Reduce bounce rate and increase time on site
   - Support content clusters and topic hubs

🔹 Internal Linking Best Practices
✓ Link to pages with topical relevance and semantic connection
✓ Use descriptive, keyword-rich anchor text (avoid "click here")
✓ Prioritize high-value pages (cornerstone content, conversion pages)
✓ Create logical user journeys (awareness → consideration → decision)
✓ Link to both parent and child pages in content hierarchy
✓ Include 3-8 internal links per page (adjust based on content length)
✓ Place links naturally within the content flow
✓ Link to both newer and older relevant content
✓ Avoid over-optimization and keyword stuffing in anchor text

🔹 Output Format

**Recommended Internal Links:**

1. **[Page Title from Sitemap]**
   - URL: [full URL]
   - Suggested Anchor Text: "[natural, keyword-rich anchor text]"
   - Placement: [Where in the content: intro, middle, conclusion, sidebar]
   - Relevance Score: [High/Medium/Low]
   - Reason: [Why this link is relevant and beneficial]

2. **[Page Title from Sitemap]**
   - URL: [full URL]
   - Suggested Anchor Text: "[natural, keyword-rich anchor text]"
   - Placement: [Where in the content]
   - Relevance Score: [High/Medium/Low]
   - Reason: [Why this link is relevant and beneficial]

[Continue for 5-10 most relevant pages]

**Content Cluster Strategy:**
- Hub Page: [Identify if this is a hub/pillar page or spoke/cluster page]
- Related Topics: [List related topics that should be interlinked]
- Missing Content: [Suggest any content gaps that should be created]

**Anchor Text Variations:**
For each recommended link, provide 2-3 alternative anchor text options to maintain natural language and avoid over-optimization.

**Implementation Priority:**
- High Priority: [Links that should be added immediately]
- Medium Priority: [Links that would be beneficial]
- Low Priority: [Optional links for comprehensive coverage]

🔹 Additional Considerations
- Analyze the sitemap structure and identify content silos
- Consider the user's journey and what they might want to read next
- Balance between SEO value and user experience
- Suggest contextual links that add value to the reader
- Identify opportunities for bidirectional linking (linking back from target pages)

🎯 Goal
Generate a strategic internal linking plan that:
✓ Improves SEO through better site structure and PageRank distribution
✓ Enhances user experience and reduces bounce rate
✓ Creates logical content pathways and topic clusters
✓ Supports conversion goals and user journey optimization

📝 Instructions for AI:
1. First, visit and analyze the sitemap at: ${sitemapUrl}
2. Extract all page URLs and titles from the sitemap
3. Match them against the current page content for relevance
4. Provide specific, actionable internal linking recommendations
5. Include exact URLs and natural anchor text suggestions

Please analyze the sitemap and provide comprehensive internal linking recommendations for this page.`;

        console.log('Internal links prompt generated successfully. Length:', prompt.length);
        return prompt;
    }

});
